% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cTMed-mc-indirect-central.R
\name{MCIndirectCentral}
\alias{MCIndirectCentral}
\title{Monte Carlo Sampling Distribution
of Indirect Effect Centrality
Over a Specific Time Interval
or a Range of Time Intervals}
\usage{
MCIndirectCentral(
  phi,
  vcov_phi_vec,
  delta_t,
  R,
  test_phi = TRUE,
  ncores = NULL,
  seed = NULL,
  tol = 0.01
)
}
\arguments{
\item{phi}{Numeric matrix.
The drift matrix (\eqn{\boldsymbol{\Phi}}).
\code{phi} should have row and column names
pertaining to the variables in the system.}

\item{vcov_phi_vec}{Numeric matrix.
The sampling variance-covariance matrix of
\eqn{\mathrm{vec} \left( \boldsymbol{\Phi} \right)}.}

\item{delta_t}{Numeric.
Time interval
(\eqn{\Delta t}).}

\item{R}{Positive integer.
Number of replications.}

\item{test_phi}{Logical.
If \code{test_phi = TRUE},
the function tests the stability
of the generated drift matrix \eqn{\boldsymbol{\Phi}}.
If the test returns \code{FALSE},
the function generates a new drift matrix \eqn{\boldsymbol{\Phi}}
and runs the test recursively
until the test returns \code{TRUE}.}

\item{ncores}{Positive integer.
Number of cores to use.
If \code{ncores = NULL},
use a single core.
Consider using multiple cores
when number of replications \code{R}
is a large value.}

\item{seed}{Random seed.}

\item{tol}{Numeric.
Smallest possible time interval to allow.}
}
\value{
Returns an object
of class \code{ctmedmc} which is a list with the following elements:
\describe{
\item{call}{Function call.}
\item{args}{Function arguments.}
\item{fun}{Function used ("MCIndirectCentral").}
\item{output}{A list the length of which is equal to
the length of \code{delta_t}.}
}
Each element in the \code{output} list has the following elements:
\describe{
\item{est}{A vector of indirect effect centrality.}
\item{thetahatstar}{A matrix of Monte Carlo
indirect effect centrality.}
}
}
\description{
This function generates a Monte Carlo method
sampling distribution
of the indirect effect centrality
at a particular time interval \eqn{\Delta t}
using the first-order stochastic differential equation model
drift matrix \eqn{\boldsymbol{\Phi}}.
}
\details{
See \code{\link[=IndirectCentral]{IndirectCentral()}} for more details.
\subsection{Monte Carlo Method}{

Let \eqn{\boldsymbol{\theta}} be
\eqn{\mathrm{vec} \left( \boldsymbol{\Phi} \right)},
that is,
the elements of the \eqn{\boldsymbol{\Phi}} matrix
in vector form sorted column-wise.
Let \eqn{\hat{\boldsymbol{\theta}}} be
\eqn{\mathrm{vec} \left( \hat{\boldsymbol{\Phi}} \right)}.
Based on the asymptotic properties of maximum likelihood estimators,
we can assume that estimators are normally distributed
around the population parameters.
\deqn{
  	\hat{\boldsymbol{\theta}}
  	\sim
  	\mathcal{N}
  	\left(
  	\boldsymbol{\theta},
  	\mathbb{V} \left( \hat{\boldsymbol{\theta}} \right)
  	\right)
  }
Using this distributional assumption,
a sampling distribution of \eqn{\hat{\boldsymbol{\theta}}}
which we refer to as \eqn{\hat{\boldsymbol{\theta}}^{\ast}}
can be generated by replacing the population parameters
with sample estimates,
that is,
\deqn{
  	\hat{\boldsymbol{\theta}}^{\ast}
  	\sim
  	\mathcal{N}
  	\left(
  	\hat{\boldsymbol{\theta}},
  	\hat{\mathbb{V}} \left( \hat{\boldsymbol{\theta}} \right)
  	\right) .
  }
Let
\eqn{\mathbf{g} \left( \hat{\boldsymbol{\theta}} \right)}
be a parameter that is a function of the estimated parameters.
A sampling distribution of
\eqn{\mathbf{g} \left( \hat{\boldsymbol{\theta}} \right)} ,
which we refer to as
\eqn{\mathbf{g} \left( \hat{\boldsymbol{\theta}}^{\ast} \right)} ,
can be generated by using the simulated estimates
to calculate
\eqn{\mathbf{g}}.
The standard deviations of the simulated estimates
are the standard errors.
Percentiles corresponding to
\eqn{100 \left( 1 - \alpha \right) \%}
are the confidence intervals.
}
}
\examples{
set.seed(42)
phi <- matrix(
  data = c(
    -0.357, 0.771, -0.450,
    0.0, -0.511, 0.729,
    0, 0, -0.693
  ),
  nrow = 3
)
colnames(phi) <- rownames(phi) <- c("x", "m", "y")
vcov_phi_vec <- matrix(
  data = c(
    0.00843, 0.00040, -0.00151,
    -0.00600, -0.00033, 0.00110,
    0.00324, 0.00020, -0.00061,
    0.00040, 0.00374, 0.00016,
    -0.00022, -0.00273, -0.00016,
    0.00009, 0.00150, 0.00012,
    -0.00151, 0.00016, 0.00389,
    0.00103, -0.00007, -0.00283,
    -0.00050, 0.00000, 0.00156,
    -0.00600, -0.00022, 0.00103,
    0.00644, 0.00031, -0.00119,
    -0.00374, -0.00021, 0.00070,
    -0.00033, -0.00273, -0.00007,
    0.00031, 0.00287, 0.00013,
    -0.00014, -0.00170, -0.00012,
    0.00110, -0.00016, -0.00283,
    -0.00119, 0.00013, 0.00297,
    0.00063, -0.00004, -0.00177,
    0.00324, 0.00009, -0.00050,
    -0.00374, -0.00014, 0.00063,
    0.00495, 0.00024, -0.00093,
    0.00020, 0.00150, 0.00000,
    -0.00021, -0.00170, -0.00004,
    0.00024, 0.00214, 0.00012,
    -0.00061, 0.00012, 0.00156,
    0.00070, -0.00012, -0.00177,
    -0.00093, 0.00012, 0.00223
  ),
  nrow = 9
)

# Specific time interval ----------------------------------------------------
MCIndirectCentral(
  phi = phi,
  vcov_phi_vec = vcov_phi_vec,
  delta_t = 1,
  R = 100L # use a large value for R in actual research
)

# Range of time intervals ---------------------------------------------------
mc <- MCIndirectCentral(
  phi = phi,
  vcov_phi_vec = vcov_phi_vec,
  delta_t = 1:5,
  R = 100L # use a large value for R in actual research
)
plot(mc)

# Methods -------------------------------------------------------------------
# MCIndirectCentral has a number of methods including
# print, summary, confint, and plot
print(mc)
summary(mc)
confint(mc, level = 0.95)
plot(mc)

}
\references{
Bollen, K. A. (1987).
Total, direct, and indirect effects in structural equation models.
Sociological Methodology, 17, 37.
\doi{10.2307/271028}

Deboeck, P. R., & Preacher, K. J. (2015).
No need to be discrete:
A method for continuous time mediation analysis.
Structural Equation Modeling: A Multidisciplinary Journal, 23 (1), 61–75.
\doi{10.1080/10705511.2014.973960}

Ryan, O., & Hamaker, E. L. (2021).
Time to intervene:
A continuous-time approach to network analysis and centrality.
Psychometrika, 87 (1), 214–252.
\doi{10.1007/s11336-021-09767-0}
}
\seealso{
Other Continuous Time Mediation Functions: 
\code{\link{BootBeta}()},
\code{\link{BootBetaStd}()},
\code{\link{BootMed}()},
\code{\link{BootMedStd}()},
\code{\link{DeltaBeta}()},
\code{\link{DeltaBetaStd}()},
\code{\link{DeltaIndirectCentral}()},
\code{\link{DeltaMed}()},
\code{\link{DeltaMedStd}()},
\code{\link{DeltaTotalCentral}()},
\code{\link{Direct}()},
\code{\link{DirectStd}()},
\code{\link{ExpCov}()},
\code{\link{ExpMean}()},
\code{\link{Indirect}()},
\code{\link{IndirectCentral}()},
\code{\link{IndirectStd}()},
\code{\link{MCBeta}()},
\code{\link{MCBetaStd}()},
\code{\link{MCMed}()},
\code{\link{MCMedStd}()},
\code{\link{MCPhi}()},
\code{\link{MCTotalCentral}()},
\code{\link{Med}()},
\code{\link{MedStd}()},
\code{\link{PosteriorBeta}()},
\code{\link{PosteriorIndirectCentral}()},
\code{\link{PosteriorMed}()},
\code{\link{PosteriorTotalCentral}()},
\code{\link{Total}()},
\code{\link{TotalCentral}()},
\code{\link{TotalStd}()},
\code{\link{Trajectory}()}
}
\author{
Ivan Jacob Agaloos Pesigan
}
\concept{Continuous Time Mediation Functions}
\keyword{cTMed}
\keyword{mc}
\keyword{network}
