test_that("clinical_data creates basic dataset correctly.", {
  result <- clinical_data()

  # Check structure and dimensions
  expect_s3_class(result, "data.frame")
  expect_equal(nrow(result), 300)  # 100 subjects * 3 visits
  expect_named(result, c("participant_id", "visit", "sex", "treatment",
                         "age", "weight", "biomarker", "response"))

  # Check data types and factor levels
  expect_type(result$participant_id, "character")
  expect_s3_class(result$sex, "factor")
  expect_equal(levels(result$sex), c("Male", "Female"))
  expect_s3_class(result$treatment, "factor")
  expect_equal(levels(result$treatment), c("Placebo", "Treatment"))
  expect_s3_class(result$response, "factor")
  expect_equal(levels(result$response), c("Complete", "Partial", "None"))
})

test_that("clinical_data handles different parameters correctly.", {
  result <- clinical_data(n = 10, visits = 5, arms = c("A", "B", "C"))
  
  expect_equal(length(unique(result$participant_id)), 10)
  expect_equal(nlevels(result$visit), 5)
  expect_equal(nrow(result), 50)  # 10 subjects * 5 visits
  expect_equal(levels(result$treatment), c("A", "B", "C"))
  expect_true(all(result$treatment %in% c("A", "B", "C")))
})

test_that("clinical_data produces reproducible results.", {
  set.seed(42)
  result1 <- clinical_data()
  set.seed(42)
  result2 <- clinical_data()
  expect_identical(result1, result2)
})

test_that("clinical_data handles dropout and missing data.", {
  # Test dropout functionality
  result_dropout <- clinical_data(dropout = 0.2)
  expect_true(any(is.na(result_dropout$biomarker)))
  expect_true(any(is.na(result_dropout$response)))
  expect_true(any(is.na(result_dropout$weight)))

  # Test missing data functionality
  result_missing <- clinical_data(missing = 0.1)
  expect_true(any(is.na(result_missing$biomarker)))
  expect_true(any(is.na(result_missing$response)))
  expect_true(any(is.na(result_missing$weight)))

  # Verify no missing data when rates are 0
  result_no_missing <- clinical_data(dropout = 0, missing = 0)
  expect_true(all(!is.na(result_no_missing[c("biomarker", "response", "weight")])))
})

test_that("clinical_data validates inputs correctly.", {
  # Test boundary conditions and invalid inputs
  expect_error(clinical_data(n = 0), "'n' must be a single integer between 1 and 999.")
  expect_error(clinical_data(n = 1000), "'n' must be a single integer between 1 and 999.")
  expect_error(clinical_data(n = 1.5), "'n' must be a single integer between 1 and 999.")
  expect_error(clinical_data(visits = 0), "'visits' must be a single positive integer.")
  expect_error(clinical_data(visits = 1.5), "'visits' must be a single positive integer.")
  expect_error(clinical_data(arms = character(0)), "'arms' must be a character vector with at least one element.")
  expect_error(clinical_data(dropout = -0.1), "'dropout' must be between 0 and 1.")
  expect_error(clinical_data(dropout = 1.1), "'dropout' must be between 0 and 1.")
  expect_error(clinical_data(missing = 1.5), "'missing' must be between 0 and 1")
  expect_error(clinical_data(visits = 1, dropout = 0.1), "Must have more than 1 visit when using 'dropout'.")
})

test_that("clinical_data maintains data integrity and shows treatment effects.", {
  result <- clinical_data(n = 50, visits = 3, arms = c("Placebo", "Low", "High"))

  # Check value ranges and format
  expect_true(all(nchar(result$participant_id) == 3))
  expect_true(all(grepl("^\\d{3}$", result$participant_id)))
  expect_true(all(result$age >= 18 & result$age <= 85))
  expect_true(all(result$weight >= 45 & result$weight <= 120))

  # Check visit sequence for first subject
  part_visits <- as.numeric(result$visit[result$participant_id == "001"])
  expect_equal(part_visits, 1:3)

  # Verify treatment effects (biomarker should decrease with treatment)
  biomarker_means <- aggregate(biomarker ~ treatment,
                               data = result[result$biomarker, ],
                               FUN = mean)
  placebo_mean <- biomarker_means$biomarker[biomarker_means$treatment == "Placebo"]
  high_mean <- biomarker_means$biomarker[biomarker_means$treatment == "High"]
  expect_gt(placebo_mean, high_mean)
})

test_that("clinical_data works with edge cases.", {
  # Single subject, single visit
  result_minimal <- clinical_data(n = 1, visits = 1)
  expect_equal(nrow(result_minimal), 1)
  expect_equal(result_minimal$participant_id, "001")

  # Maximum n
  expect_silent(clinical_data(n = 999, visits = 1))
})
