% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BIOMOD_Projection.R
\name{BIOMOD_Projection}
\alias{BIOMOD_Projection}
\title{Project a range of calibrated species distribution models onto new environment}
\usage{
BIOMOD_Projection(
  bm.mod,
  proj.name,
  new.env,
  new.env.xy = NULL,
  models.chosen = "all",
  metric.binary = NULL,
  metric.filter = NULL,
  compress = TRUE,
  build.clamping.mask = TRUE,
  nb.cpu = 1,
  seed.val = NULL,
  ...
)
}
\arguments{
\item{bm.mod}{a \code{\link{BIOMOD.models.out}} object returned by the 
\code{\link{BIOMOD_Modeling}} function}

\item{proj.name}{a \code{character} corresponding to the name (ID) of the projection set 
(\emph{a new folder will be created within the simulation folder with this name})}

\item{new.env}{a \code{matrix}, \code{data.frame} or \code{\link[raster:stack]{RasterStack}} 
object containing the new explanatory variables (in columns or layers, with names matching the 
variables names given to the \code{\link{BIOMOD_FormatingData}} function to build 
\code{bm.mod}) that will be used to project the species distribution model(s)}

\item{new.env.xy}{(\emph{optional, default} \code{NULL}) \cr 
If \code{new.env} is a \code{matrix} or a \code{data.frame}, a 2-columns \code{matrix} or 
\code{data.frame} containing the corresponding \code{X} and \code{Y} coordinates that will be 
used to project the species distribution model(s)}

\item{models.chosen}{a \code{vector} containing model names to be kept, must be either 
\code{all} or a sub-selection of model names that can be obtained with the 
\code{\link{get_built_models}} function}

\item{metric.binary}{(\emph{optional, default} \code{NULL}) \cr 
A \code{vector} containing evaluation metric names to be used to transform prediction values 
into binary values based on models evaluation scores obtained with the 
\code{\link{BIOMOD_Modeling}} function. Must be among \code{all} (same evaluation metrics than 
those of \code{bm.mod}) or \code{ROC}, \code{TSS}, \code{KAPPA}, \code{ACCURACY}, 
\code{BIAS}, \code{POD}, \code{FAR}, \code{POFD}, \code{SR}, \code{CSI}, \code{ETS}, 
\code{HK}, \code{HSS}, \code{OR}, \code{ORSS}}

\item{metric.filter}{(\emph{optional, default} \code{NULL}) \cr 
A \code{vector} containing evaluation metric names to be used to transform prediction values 
into filtered values based on models evaluation scores obtained with the 
\code{\link{BIOMOD_Modeling}} function. Must be among \code{all} (same evaluation metrics than 
those of \code{bm.mod}) or \code{ROC}, \code{TSS}, \code{KAPPA}, \code{ACCURACY}, 
\code{BIAS}, \code{POD}, \code{FAR}, \code{POFD}, \code{SR}, \code{CSI}, \code{ETS}, 
\code{HK}, \code{HSS}, \code{OR}, \code{ORSS}}

\item{compress}{(\emph{optional, default} \code{TRUE}) \cr 
A \code{logical} or a \code{character} value defining whether and how objects should be 
compressed when saved on hard drive. Must be either \code{TRUE}, \code{FALSE}, \code{xz} or 
\code{gzip} (see Details)}

\item{build.clamping.mask}{(\emph{optional, default} \code{TRUE}) \cr 
A \code{logical} value defining whether a clamping mask should be built and saved on hard 
drive or not (see Details)}

\item{nb.cpu}{(\emph{optional, default} \code{1}) \cr 
An \code{integer} value corresponding to the number of computing resources to be used to 
parallelize the single models computation}

\item{seed.val}{(\emph{optional, default} \code{NULL}) \cr 
An \code{integer} value corresponding to the new seed value to be set}

\item{\ldots}{(\emph{optional, see Details)})}
}
\value{
A \code{BIOMOD.projection.out} object containing models projections, or links to saved 
outputs. \cr Models projections are stored out of \R (for memory storage reasons) in 
\code{proj.name} folder created in the current working directory :
\enumerate{
  \item the output is a 4-dimensional array if \code{new.env} is a \code{matrix} or a 
  \code{data.frame}
  \item it is a \code{\link[raster:stack]{RasterStack}} if \code{new.env} is a 
  \code{\link[raster:stack]{RasterStack}} (or several \code{\link[raster:stack]{RasterLayer}} 
  objects, if \code{new.env} is too large)
  \item raw projections, as well as binary and filtered projections (if asked), are saved in 
  the \code{proj.name} folder
}
}
\description{
This function allows to project a range of models built with the 
\code{\link{BIOMOD_Modeling}} function onto new environmental data (\emph{which can 
represent new areas, resolution or time scales for example}).
}
\details{
If \code{models.chosen = 'all'}, projections are done for all evaluation and pseudo absences 
runs if applicable. \cr These projections may be used later by the 
\code{\link{BIOMOD_EnsembleForecasting}} function. \cr \cr 

If \code{build.clamping.mask = TRUE}, a raster file will be saved within the projection folder. 
This mask values will correspond to the number of variables in each pixel that are out of their 
calibration / training range, identifying locations where predictions are uncertain. \cr \cr

\code{...} can take the following values :
\itemize{
  \item{\code{omit.na} : }{a \code{logical} value defining whether all not fully referenced 
  environmental points will get \code{NA} as predictions or not}
  \item{\code{on_0_1000} : }{a \code{logical} value defining whether \code{0 - 1} probabilities 
  are to be converted to \code{0 - 1000} scale to save memory on backup}
  \item{\code{do.stack} : }{a \code{logical} value defining whether all projections are to be 
  saved as one \code{\link[raster:stack]{RasterStack}} object or several 
  \code{\link[raster:stack]{RasterLayer}} files (\emph{the default if projections are too heavy to 
  be all loaded at once in memory})}
  \item{\code{keep.in.memory} : }{a \code{logical} value defining whether all projections are 
  to be kept loaded at once in memory, or only links pointing to hard drive are to be returned}
  \item{\code{output.format} : }{a \code{character} value corresponding to the projections 
  saving format on hard drive, must be either \code{.grd}, \code{.img} or \code{.RData} (the 
  default if \code{new.env} is given as \code{matrix} or \code{data.frame})}
}
}
\examples{

# Load species occurrences (6 species available)
myFile <- system.file('external/species/mammals_table.csv', package = 'biomod2')
DataSpecies <- read.csv(myFile, row.names = 1)
head(DataSpecies)

# Select the name of the studied species
myRespName <- 'GuloGulo'

# Get corresponding presence/absence data
myResp <- as.numeric(DataSpecies[, myRespName])

# Get corresponding XY coordinates
myRespXY <- DataSpecies[, c('X_WGS84', 'Y_WGS84')]

# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
myFiles <- paste0('external/bioclim/current/bio', c(3, 4, 7, 11, 12), '.grd')
myExpl <- raster::stack(system.file(myFiles, package = 'biomod2'))

\dontshow{
myExtent <- raster::extent(0,30,45,70)
myExpl <- raster::stack(raster::crop(myExpl, myExtent))
}

# ---------------------------------------------------------------
file.out <- paste0(myRespName, "/", myRespName, ".AllModels.models.out")
if (file.exists(file.out)) {
  myBiomodModelOut <- get(load(file.out))
} else {

  # Format Data with true absences
  myBiomodData <- BIOMOD_FormatingData(resp.var = myResp,
                                       expl.var = myExpl,
                                       resp.xy = myRespXY,
                                       resp.name = myRespName)

  # Create default modeling options
  myBiomodOptions <- BIOMOD_ModelingOptions()

  # Model single models
  myBiomodModelOut <- BIOMOD_Modeling(bm.format = myBiomodData,
                                      modeling.id = 'AllModels',
                                      models = c('RF', 'GLM'),
                                      bm.options = myBiomodOptions,
                                      nb.rep = 2,
                                      data.split.perc = 80,
                                      metric.eval = c('TSS','ROC'),
                                      var.import = 3,
                                      do.full.models = FALSE,
                                      seed.val = 42)
}


# ---------------------------------------------------------------
# Project single models
myBiomodProj <- BIOMOD_Projection(bm.mod = myBiomodModelOut,
                                  proj.name = 'Current',
                                  new.env = myExpl,
                                  models.chosen = 'all',
                                  metric.binary = 'all',
                                  metric.filter = 'all',
                                  build.clamping.mask = TRUE)
myBiomodProj
plot(myBiomodProj)


}
\seealso{
\code{\link{BIOMOD_Modeling}}, \code{\link{BIOMOD_EnsembleModeling}}, 
\code{\link{BIOMOD_RangeSize}}

Other Main functions: 
\code{\link{BIOMOD_CrossValidation}()},
\code{\link{BIOMOD_EnsembleForecasting}()},
\code{\link{BIOMOD_EnsembleModeling}()},
\code{\link{BIOMOD_FormatingData}()},
\code{\link{BIOMOD_LoadModels}()},
\code{\link{BIOMOD_ModelingOptions}()},
\code{\link{BIOMOD_Modeling}()},
\code{\link{BIOMOD_PresenceOnly}()},
\code{\link{BIOMOD_RangeSize}()},
\code{\link{BIOMOD_Tuning}()}
}
\author{
Wilfried Thuiller, Damien Georges
}
\concept{Main functions}
\keyword{models}
\keyword{projection}
