% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bigleaf_physiology.r
\name{photosynthetic.capacity}
\alias{photosynthetic.capacity}
\title{Bulk Canopy Photosynthetic Capacity (Vcmax and Jmax)}
\usage{
photosynthetic.capacity(data, C3 = TRUE, Temp, GPP = "GPP", Ci,
  PPFD = "PPFD", PPFD_j = c(200, 500), PPFD_c = 1000, Rleaf = NULL,
  Oi = 0.21, Kc25 = 404.9, Ko25 = 278.4, Gam25 = 42.75, Kc_Ha = 79.43,
  Ko_Ha = 36.38, Gam_Ha = 37.83, Vcmax_Ha = 65.33, Vcmax_Hd = 200,
  Vcmax_dS = 0.635, Jmax_Ha = 43.9, Jmax_Hd = 200, Jmax_dS = 0.64,
  Theta = 0.7, alpha_canopy = 0.8, missing.Rleaf.as.NA = FALSE,
  Ci_C4 = 100, constants = bigleaf.constants())
}
\arguments{
\item{data}{Data.Frame or matrix with all required columns}

\item{C3}{C3 vegetation (\code{TRUE}, the default) or C4 vegetation (\code{FALSE})?}

\item{Temp}{Surface (or air) temperature (degC)}

\item{GPP}{Gross primary productivity (umol m-2 s-1)}

\item{Ci}{Bulk canopy intercellular CO2 concentration (umol mol-1)}

\item{PPFD}{Photosynthetic photon flux density (umol m-2 s-1)}

\item{PPFD_j}{PPFD threshold, below which the canopy is considered to 
be RuBP regeneration limited. Defaults to 500 umol m-2 s-1.}

\item{PPFD_c}{PPFD threshold, above which the canopy is considered to 
be Rubisco limited. Defaults to 1000 umol m-2 s-1.}

\item{Rleaf}{Ecosystem respiration stemming from leaves (umol CO2 m-2 s-1); defaults to 0}

\item{Oi}{Intercellular O2 concentration (mol mol-1)}

\item{Kc25}{Michaelis-Menten constant for CO2 at 25 degC (umol mol-1)}

\item{Ko25}{Michaelis-Menten constant for O2 at 25 degC (mmol mol-1)}

\item{Gam25}{Photorespiratory CO2 compensation point ('Gamma star') 
at 25 degC (umol mol-1)}

\item{Kc_Ha}{Activation energy for Kc (kJ mol-1)}

\item{Ko_Ha}{Activation energy for Ko (kJ mol-1)}

\item{Gam_Ha}{Activation energy for Gam (kJ mol-1)}

\item{Vcmax_Ha}{Activation energy for Vcmax (kJ mol-1)}

\item{Vcmax_Hd}{Deactivation energy for Vcmax (kJ mol-1)}

\item{Vcmax_dS}{Entropy term for Vcmax (kJ mol-1 K-1)}

\item{Jmax_Ha}{Activation energy for Jmax (kJ mol-1)}

\item{Jmax_Hd}{Deactivation energy for Jmax (kJ mol-1)}

\item{Jmax_dS}{Entropy term for Jmax (kJ mol-1 K-1)}

\item{Theta}{Curvature term in the light response function of J (-)}

\item{alpha_canopy}{Canopy absorptance (-)}

\item{missing.Rleaf.as.NA}{if Rleaf is provided, should missing values be treated as \code{NA} (\code{TRUE})
or set to 0 (\code{FALSE}, the default)?}

\item{Ci_C4}{intercellular CO2 concentration below which photosynthesis
is considered to be CO2-limited (umol mol-1), ignored
if \code{C3 = TRUE}.}

\item{constants}{Kelvin - conversion degree Celsius to Kelvin \cr
Rgas - universal gas constant (J mol-1 K-1)}
}
\value{
a data.frame with the following columns:
        \item{Vcmax25}{maximum bulk canopy carboxylation rate at 25degC (umol m-2 (ground) s-1)}
        \item{Jmax25}{maximum bulk canopy electron transport rate at 25degC (umol m-2 (ground) s-1)}
}
\description{
Bulk canopy maximum carboxylation rate (Vcmax25), and maximum electron
             transport rate (Jmax25) at 25 degrees Celsius from bulk intercellular 
             CO2 concentration using the Farquhar et al. 1980 model for C3 photosynthesis.
}
\details{
The maximum carboxylation rate at 25degC (Vcmax25) and the maximum electron
         transport rate at 25degC (Jmax25), which characterize photosynthetic capacity,
         are calculated as at leaf level. 
         The required variables Gs and Ci can be calculated from 
         \code{\link{surface.conductance}} and \code{\link{intercellular.CO2}}, respectively.
         
         Gas exchange parameters are taken from Bernacchi et al. 2001 (apparent values, which
         assume an infinite mesophyll conductance). Negative and very low Ci values 
         (the threshold is set to Ci < 80umol mol-1 at the moment) are filtered out.
         
         Vcmax is calculated from the photosynthesis model by Farquhar et al. 1980.
         If net photosynthesis is Rubisco-limited (RuBP-saturated carboxylation
         rate, i.e. light has to be (near-)saturating):
        
           \deqn{Vcmax = (GPP * (Ci + Kc*(1.0 + Oi/Ko))) / (Ci - Gam)}
         
         where Kc and Ko are the Michaelis-Menten constants for CO2 and O2 (mmol mol-1),
         respectively, Oi is the O2 concentration, and Gam is the photorespiratory CO2
         compensation point (umol mol-1).
         Under low-light conditions, the electron transport rate J is calculated from
         the RuBP regeneration-limited photosynthesis rate:
         
           \deqn{J = (GPP * (4.0 * Ci + 8.0 * Gam) / (Ci - Gam)}
         
         In this function, bulk canopy photosynthesis is assumed to be Rubisco/RuBP-regeneration
         limited, if incoming PPFD is above/below a specified threshold or range. These ranges
         are determined by the parameters \code{PPFD_j} and \code{PPFD_c}. If, for example,
         \code{PPFD_j = c(100,400)}, all conditions with a PPFD between 100 and 400 are assumed
         to be in the RuBP-regeneration (i.e. light-limited) photosynthesis domain. The electron
         transport rate J is then only calculated for periods that meet this criterion.
         
         Jmax is calculated from J and absorbed irradiance:
         
           \deqn{J = (APPFD_PSII + Jmax - sqrt((APPFD_PSII + Jmax)^2 - 
                    4.0 * Theta * APPFD_PSII * Jmax)) / (2.0 * Theta)
                }
              
         where APPFD_PSII is the absorbed PPFD by photosystem II (PS II), 
         and Theta is a curvature parameter. APPFD_PSII is calculated as
         
           \deqn{PPFD * alpha_canopy * 0.85 * beta}
         
         where alpha_canopy is canopy-scale absorptance, 0.85 is a correction factor,
         and beta is the fraction of photons absorbed by PS II (assumed 0.5).
         alpha_canopy accounts for non-absorbing components of the ecosystem such as
         stems or soil, and is very likely ecosystem-specific. This parameter is relatively
         sensitive for the determination of Jmax25 at some sites.
         
         Vcmax and Jmax at canopy level are assumed to follow the same temperature response
         as at leaf level. Hence, the respective parameter k at 25degC (k25) is calculated as 
         (see e.g. Kattge & Knorr 2007):
         
           \deqn{k25 = k / 
                       ( exp(Ha * (Temp - Tref) / (Tref * Rgas * Temp)) *
                       (1 + exp((Tref * dS - Hd) / (Tref * Rgas))) /
                       (1 + exp((Temp * dS - Hd) / (Temp * Rgas)))
                       )
                 }
         
         where Ha is the activation energy (kJ mol-1), Hd is the deactivation energy (kJ mol-1),
         and dS is the entropy term (kJ mol-1 K-1) of the respective parameter. Tref is set
         to 298.15 K.
         
         For C4 photosynthesis, the simplified model by von Caemmerer 2000 is used.
         For light-saturated photosynthesis, Vcmax is given by:
         
           \deqn{Vcmax = GPP}
         
         Note that in addition to the range \code{PPFD_c}, the range \code{Ci_C4}
         discards all periods with low Ci, in which photosynthesis is likely to
         be CO2-limited (see von Caemmerer 2000 for details).
         
         In the light-limited case, J is calculated as:
         
           \deqn{J = 3 * GPPj / (1 - 0.5) }
         
         The calculation of Jmax25 and Vcmax25 is identical to C3 photosynthesis
         as described above.
}
\note{
The critical assumption is that bulk canopy photosynthesis is limited by
        one of the two limitation states. Incoming PPFD is assumed to determine
        the limitation states. Note however that the ranges (\code{PPFD_j} and \code{PPFD_c})
        are likely ecosystem-specific. E.g. dense canopies presumably require higher
        \code{PPFD_c} thresholds than open canopies. A threshold of 500 umol m-2 s-1 PPFD
        for Rubisco-limited photosynthesis was assumed a reasonable working assumption (see Kosugi et al. 2013).
        Here, \code{PPFD_c} defaults to 1000 umol m-2 s-1. Note that even under very high/low irradiances,
        not all photosynthetically active plant material of an ecosystem will be in the same
        limitation state. Note that parameters describing bulk canopy photosynthetic capacity are not directly 
        comparable to their leaf-level counterparts, as the former integrate over the entire canopy
        depth (i.e. are given per ground area, and not per leaf area).
        In general, the function should be used with care!
}
\examples{
DE_Tha_Jun_2014_2 <- filter.data(DE_Tha_Jun_2014,quality.control=FALSE,
                                 vars.qc=c("Tair","precip","VPD","H","LE"),
                                 filter.growseas=FALSE,filter.precip=TRUE,
                                 filter.vars=c("Tair","PPFD","ustar","LE"),
                                 filter.vals.min=c(5,200,0.2,0),
                                 filter.vals.max=c(NA,NA,NA,NA),NA.as.invalid=TRUE,
                                 quality.ext="_qc",good.quality=c(0,1),
                                 missing.qc.as.bad=TRUE,GPP="GPP",doy="doy",
                                 year="year",tGPP=0.5,ws=15,min.int=5,precip="precip",
                                 tprecip=0.1,precip.hours=24,records.per.hour=2)

# calculate Ga
Ga <- aerodynamic.conductance(DE_Tha_Jun_2014_2,Rb_model="Thom_1972")[,"Ga_h"]

# calculate Gs from the the inverted PM equation
Gs_PM <- surface.conductance(DE_Tha_Jun_2014_2,Tair="Tair",pressure="pressure",
                             Rn="Rn",G="G",S=NULL,VPD="VPD",Ga=Ga,
                             formulation="Penman-Monteith")[,"Gs_mol"]

# calculate Ci 
Ci <- intercellular.CO2(DE_Tha_Jun_2014_2,Ca="Ca",GPP="GPP",Gs=Gs_PM) 

# calculate Vcmax25 and Jmax25
photosynthetic.capacity(DE_Tha_Jun_2014_2,Temp="Tair",Ci=Ci,PPFD_j=c(200,500),PPFD_c=1000)

}
\references{
Lloyd J. et al., 1995: A simple calibrated model of Amazon rainforest productivity
            based on leaf biochemical properties. Plant, Cell and Environment 18, 1129-1145.

            Rayment M.B., Loustau D., Jarvis P.G., 2002: Photosynthesis and respiration
            of black spruce at three organizational scales: shoot, branch and canopy.
            Tree Physiology 22, 219-229.

            Kosugi Y. et al., 2013: Determination of the gas exchange phenology in an
            evergreen coniferous forest from 7 years of eddy covariance flux data using
            an extended big-leaf analysis. Ecol Res 28, 373-385. 
            
            Ueyama M. et al, 2016: Optimization of a biochemical model with eddy covariance
            measurements in black spruce forests of Alaska for estimating CO2 fertilization
            effects. Agricultural and Forest Meteorology 222, 98-111.
            
            Bernacchi C.J., Singsaas E.L., Pimentel C., Portis JR A.R., Long S.P., 2001:
            Improved temperature response functions for models of Rubisco-limited
            photosynthesis. Plant, Cell and Environment 24, 253-259. 
            
            Bernacchi C.J., Pimentel C., Long S.P., 2003: In vivo temperature response
            functions of parameters required to model RuBP-limited photosynthesis.
            Plant, Cell and Environment 26, 1419-1430.
            
            von Caemmerer, 2000: Biochemical models of leaf photosynthesis. Techniques
            in plant sciences No. 2. CSIRO Publishing, Collingwood VIC, Australia.
}
\seealso{
\code{\link{intercellular.CO2}}, \code{\link{Arrhenius.temp.response}}
}
