% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/decoupling.r
\name{decoupling}
\alias{decoupling}
\title{Canopy-Atmosphere Decoupling Coefficient}
\usage{
decoupling(data, Tair = "Tair", pressure = "pressure", Ga = "Ga",
  Gs = "Gs", approach = c("JarvisMcNaughton_1986", "Martin_1989"), LAI,
  constants = bigleaf.constants())
}
\arguments{
\item{data}{Data.frame or matrix containing all required input variables}

\item{Tair}{Air temperature (deg C)}

\item{pressure}{Atmospheric pressure (kPa)}

\item{Ga}{Aerodynamic conductance (m s-1)}

\item{Gs}{Surface conductance (m s-1)}

\item{approach}{Approach used to calculate omega. Either \code{"JarvisMcNaughton_1986"} (default)
or \code{"Martin_1989"}.}

\item{LAI}{Leaf area index (m2 m-2), only used if \code{approach = "Martin_1989"}.}

\item{constants}{Kelvin - conversion degree Celsius to Kelvin \cr
cp - specific heat of air for constant pressure (J K-1 kg-1) \cr
eps - ratio of the molecular weight of water vapor to dry air (-) \cr
sigma - Stefan-Boltzmann constant (W m-2 K-4) \cr}
}
\value{
\item{\eqn{\Omega} -}{the decoupling coefficient Omega (-)}
}
\description{
The canopy-atmosphere decoupling coefficient 'Omega'.
}
\details{
The decoupling coefficient Omega ranges from 0 to 1 and quantifies the
         linkage of the conditions (foremost humidity and temperature) at the canopy surface
         to the ambient air. Values close to 0 indicate well coupled conditions
         characterized by high physiological (i.e. stomatal) control on transpiration
         and similar conditions at the canopy surface compared to the atmosphere above
         the canopy. Values close to 1 indicate the opposite, i.e. decoupled conditions and 
         a low stomatal control on transpiration (Jarvis & McNaughton 1986). \cr
         The \code{"JarvisMcNaughton_1986"} approach (default option) is the original
         formulation for the decoupling coefficient, given by (for an amphistomatous 
         canopy):
         
         \deqn{\Omega = \frac{\epsilon + 1}{\epsilon + 1 + \frac{Ga}{Gc}}}{%
         \Omega = (\epsilon + 1) / ( \epsilon + 1 + Ga/Gc)}
         
         where \eqn{\epsilon = \frac{s}{\gamma}}{\epsilon = s/\gamma} is a dimensionless coefficient
         with s being the slope of the saturation vapor pressure curve (Pa K-1), and \eqn{\gamma} the 
         psychrometric constant (Pa K-1).
         
         The approach \code{"Martin_1989"} by Martin 1989 additionally takes radiative coupling
         into account:
         
         \deqn{\Omega = \frac{\epsilon + 1 + \frac{Gr}{Ga}}{\epsilon + (1 + \frac{Ga}{Gs}) (1 + \frac{Gr}{Ga})}}{%
         \Omega = (\epsilon + 1 + Gr/Ga) / (\epsilon + (1 + Ga/Gs) (1 + Gr/Ga))}
}
\examples{
# Omega calculated following Jarvis & McNaughton 1986
set.seed(3)
df <- data.frame(Tair=rnorm(20,25,1),pressure=100,Ga=rnorm(20,0.06,0.01),Gs=rnorm(20,0.005,0.001))
decoupling(df,approach="JarvisMcNaughton_1986")

# Omega calculated following Martin 1989 (requires LAI)
decoupling(df,approach="Martin_1989",LAI=4)

}
\references{
Jarvis P.G., McNaughton K.G., 1986: Stomatal control of transpiration:
            scaling up from leaf to region. Advances in Ecological Research 15, 1-49. 
            
            Martin P., 1989: The significance of radiative coupling between
            vegetation and the atmosphere. Agricultural and Forest Meteorology 49, 45-53.
}
\seealso{
\code{\link{aerodynamic.conductance}}, \code{\link{surface.conductance}},
         \code{\link{equilibrium.imposed.ET}}
}
