#' Estimate Semipartial Correlation Coefficients
#' and Generate the Corresponding Sampling Distribution
#' Using the Monte Carlo Method
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details The vector of semipartial correlation coefficients
#'   (\eqn{r_{s}})
#'   is derived from each randomly generated vector of parameter estimates.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{r_{s}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betamc` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{object}{The function argument `object`.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{r_{s}}.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{r_{s}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{r_{s}}.}
#'     \item{fun}{Function used ("SCorMC").}
#'   }
#'
#' @inheritParams BetaMC
#'
#' @examples
#' # Fit the regression model
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#' # Generate the sampling distribution of parameter estimates
#' # (use a large R, for example, R = 20000 for actual research)
#' mc <- MC(object, R = 100)
#' # Generate confidence intervals for standardized regression slopes
#' rs <- SCorMC(mc)
#' # Methods --------------------------------------------------------
#' print(rs)
#' summary(rs)
#' coef(rs)
#' vcov(rs)
#' confint(rs, level = 0.95)
#' @export
#' @family Beta Monte Carlo Functions
#' @keywords betaMC scor
SCorMC <- function(object) {
  stopifnot(
    methods::is(
      object,
      "mc"
    )
  )
  if (object$lm_process$p < 2) {
    stop("Two or more regressors is required.")
  }
  std <- BetaMC(object)
  std <- std$thetahatstar
  thetahatstar <- mapply(
    theta = object$thetahatstar,
    std = std,
    SIMPLIFY = FALSE,
    FUN = function(theta,
                   std) {
      return(
        .SPCor(
          betastar = std,
          sigmacapx = theta$sigmacapx
        )
      )
    }
  )
  est <- .SPCor(
    betastar = object$lm_process$betastar,
    sigmacapx = object$lm_process$sigmacapx
  )
  names(est) <- object$lm_process$xnames
  out <- list(
    call = match.call(),
    object = object,
    thetahatstar = thetahatstar,
    est = est,
    fun = "SCorMC"
  )
  class(out) <- c(
    "betamc",
    class(out)
  )
  return(
    out
  )
}
