#' Calculate effective number of parameters
#'
#' \code{p_waic} calculates effective number of parameters. This is
#'   used as a bias adjustment factor in the calculation of WAIC
#'   in the function \code{waic}.
#'
#'   NOTE: in order to calculated pWAIC, the model MUST track the parameter
#'   "lambda". In species that are data-rich, such as Wood Thrush,
#'   this produces extremely large JAGS objects, and takes up a considerable
#'   amount of memory when simulating with \code{run_model}
#'
#' @param jags_data Data prepared by \code{prepare_jags_data}, used
#'   for input to the JAGS model
#' @param jags_mod JAGS list generated by \code{run_model}
#' @param pointwise If set to \code{TRUE}, a data frame is returned
#'   that contains the pointwise LPPD for each count. Defaults
#'   to \code{FALSE}
#'
#' @return Data frame of pointwise pWAIC by count if \code{pointwise}
#'   is set to \code{TRUE}. Double precision numerical value of pWAIC if
#'   \code{pointwise} is set to \code{FALSE}.
#'
#' @importFrom stats dpois var
#'
#'
#' @name bbsBayes-deprecated
#' @seealso \code{\link{p_waic}}
#' @keywords internal
NULL

#' @rdname bbsBayes-deprecated
#' @section \code{p_waic}:
#'   WAIC should no longer be used for BBS data.
#'   Cross validation should be used instead.
#'
#' @export
#'
p_waic <- function(jags_data = NULL,
                   jags_mod = NULL,
                   pointwise = FALSE)
{
  .Deprecated(msg = "WAIC should no longer be used for BBS data. Use cross validation instead.")

  bugs <- get_prepared_data(jags_data)

  pwaic_point <- data.frame(index = 1:nrow(bugs))
  pwaic_v <- vector(length = nrow(bugs))

  for(i in 1:nrow(bugs))
  {
    pwaic_v[i] <- stats::var(log(stats::dpois(bugs[i,"Count"], jags_mod$sims.list$lambda[,i])))
  }

  pwaic_point[,"pwaic_point"] <- pwaic_v

  if (isTRUE(pointwise))
  {
    return(pwaic_point)
  }else{
    return(sum(pwaic_point$pwaic_point))
  }
}
