% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_funs.R
\name{eDNA_lm}
\alias{eDNA_lm}
\alias{eDNA_lmer}
\alias{eDNA_zinf_lm}
\alias{eDNA_zinf_lmer}
\title{Fit eDNA Model}
\usage{
eDNA_lm(
  formula,
  data,
  std_curve_alpha,
  std_curve_beta,
  upper_Cq = 40,
  prior_intercept = normal(location = -15, scale = 10),
  priors = normal(),
  ...
)

eDNA_lmer(
  formula,
  data,
  std_curve_alpha,
  std_curve_beta,
  upper_Cq = 40,
  prior_intercept = normal(location = -15, scale = 10),
  priors = normal(),
  prior_random_variance = exponential(),
  ...
)

eDNA_zinf_lm(
  formula,
  data,
  std_curve_alpha,
  std_curve_beta,
  upper_Cq = 40,
  probability_zero = 0.08,
  prior_intercept = normal(location = -15, scale = 10),
  priors = normal(),
  ...
)

eDNA_zinf_lmer(
  formula,
  data,
  std_curve_alpha,
  std_curve_beta,
  upper_Cq = 40,
  probability_zero = 0.08,
  prior_intercept = normal(location = -15, scale = 10),
  priors = normal(),
  prior_random_variance = exponential(),
  ...
)
}
\arguments{
\item{formula}{a formula, specifying the relationship between the
predictors and the latent variable eDNA concentration.}

\item{data}{data.frame, with the response and predictors}

\item{std_curve_alpha}{the alpha value for the formula for
converting between log(eDNA concentration) and CQ value}

\item{std_curve_beta}{the beta value for the formula for
converting between log(eDNA concentration) and CQ value}

\item{upper_Cq}{numeric, the upper limit on CQ detection. Any
value of log(concentration) which would result in a value
greater than this limit is instead recorded as the limit.}

\item{prior_intercept}{named list such as created by
\code{rstanarm::normal}. The list must contain elements named
"location" and "scale", which are the location and scale for a
normal prior over the intercept. Ignored when the intercept is
omitted in the model formula.}

\item{priors}{named list such as created by
\code{rstanarm::normal}. The list must contain elements named
"location" and "scale", which are the location and scale for a
normal prior over the betas, and "autoscale". If a single
value is provided, this value will be repeated for each
beta. If \code{autoscale = TRUE}, the scale of the priors is
scaled by the sd of the predictors similar to rstanarm handles
them.}

\item{...}{additional arguments passed to
\code{\link[rstan]{sampling}}}

\item{prior_random_variance}{the prior on variance of the random
effects. Defaults to exponential distribution with rate 1.}

\item{probability_zero}{numeric, between 0 and 1. The probability
of a non-detection from a source other than low concentration
of eDNA, e.g. a filter failure. Defaults to 8% (0.08), which
was the estimated p(zero) from a daily sampling experiment.}
}
\value{
S4 object, with the following slots:
\describe{
  \item{ln_conc}{matrix, the posterior samples for the latent
    variable, eDNA concentration}
\item{Cq_star}{matrix, the
    posterior prediction for the observed response}
\item{betas}{array, the posterior estimates for the betas for
    the linear model}
\item{sigma_ln_eDNA}{array, the posterior
    estimates for the measurement error of ln_eDNA}
\item{formula}{formula, the original formula used in the
    model}
\item{x}{data.frame, the model matrix used in the
    model}
\item{std_curve_alpha}{numeric, the std. curve value
    used}
\item{std_curve_beta}{numeric, the std. curve value
    used}
\item{upper_Cq}{numeric, the upper limit for observed CQ
    used}
\item{stanfit}{stanfit, the original results from
    \code{rstan::sampling}} }
}
\description{
Fit eDNA model
}
\details{
These functions fit a Bayesian latent variable model to data
collected from a eDNA sampling experiment. These data have a few
particular characteristics that justify using a specialized
model. More details on these characteristics and the model
structure, please refer to the "Getting Started" vignette for the
artemis package.

There are four different modeling functions in the artemis
package, \code{eDNA_lm}, \code{eDNA_lmer}, \code{eDNA_zinf_lm},
\code{eDNA_zinf_lmer}.  \code{eDNA_lm} is for fitting a fixed
effects model, while \code{eDNA_lmer} is for fitting a mixed or
random effects model. The *_zinf versions implement a
zero-inflated version of their respective lm function. All models
are fit using the \code{rstan::sampling} function, which uses a
Hamiltonian Monte Carlo algorithm to estimate parameters for the
model. Users are encouraged to refer to the documentation for Stan
and RStan at \url{https://mc-stan.org/users/documentation/} for
details about how models are fit.
}
\section{Diagnosing warning and error messages}{


The models have been written in Stan with key focus on robustness
and speed. However, it is possible that users might encounter
issues. Typically, these issues will be highlighted by warning
messages coming from \code{rstan::sampling}. Often times, these
warnings can be resolved by increasing the number of iterations
that the HMC algorithm runs by specifying \code{iters} to be a
larger value. This should be the first action attempted, as
increasing the \code{iters} increases both the warm-up and
sampling iterations. If users continue to have issues, additional
control arguments can be passed to \code{rstan::sampling} via the
\code{...} argument.
}

\examples{

## Fixed effect model
ans = eDNA_lm(Cq ~ Distance_m, eDNA_data,
              std_curve_alpha = 21.2, std_curve_beta = -1.5)

\donttest{
## Mixed-effect model
## This takes a while to run
ans2 = eDNA_lmer(Cq ~ Distance_m + (1|FilterID), eDNA_data,
                 std_curve_alpha = 21.2, std_curve_beta = -1.5)

}

}
\author{
Matt Espe
}
