% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/z_animint.R
\name{animint2dir}
\alias{animint2dir}
\title{Compile and render an animint in a local directory.}
\usage{
animint2dir(plot.list, out.dir = NULL, json.file = "plot.json",
  open.browser = interactive(), css.file = "")
}
\arguments{
\item{plot.list}{a named list of ggplots and option lists.}

\item{out.dir}{directory to store html/js/csv files. If it exists
already, it will be removed before writing the new
directory/files.}

\item{json.file}{character string that names the JSON file with
metadata associated with the plot.}

\item{open.browser}{Should R open a browser? If yes, be sure to
configure your browser to allow access to local files, as some
browsers block this by default (e.g. chrome).}

\item{css.file}{character string for non-empty css file to
include. Provided file will be copied to the output directory as
styles.css}
}
\value{
invisible list of ggplots in list format.
}
\description{
This function converts an animint plot.list into a directory of
files which can be used to render the interactive data
visualization in a web browser.
}
\examples{
## Make a Gapminder plot (aka Google motion chart), which is actually
## just a scatterplot with size and color that moves over time.
library(animint2)
data(WorldBank)
gapminder <-
  list(title="Linked scatterplot and time series",
       ts=ggplot()+
       make_tallrect(WorldBank, "year")+
       geom_line(aes(year, life.expectancy, group=country, color=region),
                 clickSelects="country",
                 data=WorldBank, size=4, alpha=3/5),
       time=list(variable="year",ms=3000),
       duration=list(year=1000),
       scatter=ggplot()+
       geom_point(aes(fertility.rate, life.expectancy,
                      key=country, colour=region, size=population),
                  showSelected="year",
                  clickSelects="country",
                  data=WorldBank)+
       geom_text(aes(fertility.rate, life.expectancy, label=country),
                 showSelected=c("country", "year"),
                 data=WorldBank)+
       make_text(WorldBank, 5, 80, "year")+
       scale_size_animint(pixel.range=c(2,20), breaks=10^(4:9)))
animint2dir(gapminder)

data(worldPop)
## Linked bar and line plots of world population by subcontinent,
## inspired by polychartjs.
popPlots <-
  list(bars=ggplot()+
       geom_bar(aes(x=subcontinent, y=population),
                clickSelects="subcontinent",
                showSelected="year",
                data=worldPop, stat="identity", position="identity")+
       ## This make_text creates a geom_text that shows the current
       ## selected value of the year variable.
       make_text(worldPop, 1, 3e6, "year")+
       coord_flip(),
       lines=ggplot()+
       ## This make_tallrect tiles the background of the lineplot with
       ## rects that can be clicked to select the year variable.
       make_tallrect(worldPop, "year")+
       ## This geom_point does not have aes(clickSelects) so its alpha
       ## transparency behaves normally: all points have alpha=1/4.
       geom_point(aes(year, population, colour=type),
                  data=worldPop, size=4, alpha=1/4)+
       ## This geom_line DOES have aes(clickSelects) so only the
       ## selected line has the specified alpha=3/4. The other
       ## unselected lines have 0.5 less (alpha=1/4).
       geom_line(aes(year, population, group=subcontinent),
                 clickSelects="subcontinent",
                 data=worldPop, size=4, alpha=3/4))
animint2dir(popPlots)
## Make it animated by specifying year as the variable to animate and
## an interval of 2000 milliseconds between animation frames.
popAnim <- c(popPlots, list(time=list(variable="year",ms=2000)))
animint2dir(popAnim)
## Make the animation smooth by specifying a duration of 1000 ms for
## geoms with aes(showSelected=year).
popSmooth <- c(popAnim, list(duration=list(year=1000)))
animint2dir(popSmooth)

}
