##' A wrapper for the `convert' utility of ImageMagick or GraphicsMagick.
##' The main purpose of these two functions is to create GIF animations.
##'
##' the function \code{im.convert} simply wraps the arguments of the
##' \command{convert} utility of ImageMagick to make it easier to call
##' ImageMagick in R;
##'
##' @aliases im.convert gm.convert
##' @rdname convert
##' @param files either a character vector of file names, or a single
##' string containing wildcards (e.g. \file{Rplot*.png})
##' @param output the file name of the output (with proper extensions,
##' e.g. \code{gif})
##' @param convert the \command{convert} command; it must be either
##' \code{'convert'} or \code{'gm convert'}; and it can be pre-specified
##' as an option in \code{\link{ani.options}('convert')},
##' e.g. (Windows users) \code{ani.options(convert =
##' shQuote('c:/program files/imagemagick/convert.exe'))}, or (Mac
##' users) \code{ani.options(convert = '/opt/local/bin/convert')}; see
##' the Note section for more details
##' @param cmd.fun a function to invoke the OS command; by default
##' \code{\link[base]{system}}
##' @param extra.opts additional options to be passed to
##' \command{convert} (or \command{gm convert})
##' @param clean logical: delete the input \code{files} or not
##' @return The path of the output if the command was successfully
##' executed; otherwise a failure message.
##'
##' If \code{ani.options('autobrowse') == TRUE}, this function will
##' also try to open the output automatically.
##' @note If \code{files} is a character vector, please make sure the
##' order of filenames is correct! The first animation frame will be
##' \code{files[1]}, the second frame will be \code{files[2]}, ...
##'
##' Most Windows users do not have read the boring notes below after
##' they have installed ImageMagick or GraphicsMagick. For the rest of
##' Windows users:
##'
##' \describe{
##'
##' \item{\strong{ImageMagick users}}{Please install ImageMagick from
##' \url{http://www.imagemagick.org}, and make sure the the path to
##' \command{convert.exe} is in your \code{'PATH'} variable, in which
##' case the command \command{convert} can be called without the full
##' path.  Windows users are often very confused about the ImageMagick
##' and \code{'PATH'} setting, so I'll try to search for ImageMagick
##' in the Registry Hive by
##' \code{readRegistry('SOFTWARE\ImageMagick\Current')$BinPath}, thus
##' you might not really need to modify your \code{'PATH'} variable.
##'
##' For Windows users who have installed LyX, I will also try to find
##' the \command{convert} utility in the LyX installation directory,
##' so they do not really have to install ImageMagick if LyX exists in
##' their system (of course, the LyX should be installed with
##' ImageMagick).
##'
##' Once the \command{convert} utility is found, the animation option
##' \code{'convert'} will be set (\code{ani.options(convert =
##' 'path/to/convert.exe')}); this can save time for searching for
##' \command{convert} in the operating system next time.  }
##'
##' \item{\strong{GraphicsMagick users}}{During the installation of
##' GraphicsMagick, you will be asked if you allow it to change the
##' PATH variable; please do check the option.  }
##'
##' }
##'
##' A reported problem is \code{cmd.fun = shell} might not work under
##' Windows but \code{cmd.fun = system} works fine. Try this option in
##' case of failures.
##' @author Yihui Xie <\url{http://yihui.name}>
##' @seealso \code{\link{saveGIF}}
##' @references
##' ImageMagick: \url{http://www.imagemagick.org/script/convert.php}
##'
##' GraphicsMagick: \url{http://www.graphicsmagick.org}
##' @examples
##' ## generate some images
##' owd = setwd(tempdir())
##' oopt = ani.options(interval = 0.05, nmax = 20)
##' png("bm%03d.png")
##' brownian.motion(pch = 21, cex = 5, col = "red", bg = "yellow",
##' main = "Demonstration of Brownian Motion")
##' dev.off()
##'
##' ## filenames with a wildcard *
##' im.convert("bm*.png", output = "bm-animation1.gif")
##' ## use GraphicsMagick
##' gm.convert("bm*.png", output = "bm-animation2.gif")
##'
##' ## or a filename vector
##' bm.files = sprintf("bm%03d.png", 1:20)
##' im.convert(files = bm.files, output = "bm-animation3.gif")
##'
##' ani.options(oopt)
##' setwd(owd)
##'
im.convert = function(files, output = "animation.gif", convert = c("convert",
                                                       "gm convert"),
                      cmd.fun = system, extra.opts = "", clean = FALSE) {
    output.path = file.path(ani.options('outdir'), output)
    interval = head(ani.options('interval'), length(files))
    convert = match.arg(convert)
    if (convert == 'convert') {
        version = ''
        if (!is.null(ani.options('convert'))) {
            version = try(cmd.fun(sprintf("%s --version", ani.options('convert')),
                              intern = TRUE, ignore.stdout = .ani.env$check, ignore.stderr = .ani.env$check))
        }
        if (!length(grep("ImageMagick", version))) {
            version = try(cmd.fun(sprintf("%s --version", convert), intern = TRUE,
                                  ignore.stdout = .ani.env$check, ignore.stderr = .ani.env$check))
        } else convert = ani.options('convert')
        ## try to look for ImageMagick in the Windows Registry Hive,
        ## the Program Files directory and the LyX installation
        if (!length(grep("ImageMagick", version))) {
            message("I cannot find ImageMagick with convert = ", shQuote(convert))
            if (.Platform$OS.type == "windows") {
                if (!inherits(try({
                    magick.path = utils::readRegistry("SOFTWARE\\ImageMagick\\Current")$BinPath
                }, silent = TRUE), "try-error")) {
                    if (nzchar(magick.path)) {
                        convert = shQuote(normalizePath(file.path(magick.path,
                        "convert.exe")))
                        message("but I can find it from the Registry Hive: ",
                                magick.path)
                    }
                }
                else if (nzchar(prog <- Sys.getenv("ProgramFiles")) &&
                         length(magick.dir <- list.files(prog, "^ImageMagick.*")) &&
                         length(magick.path <- list.files(file.path(prog,
                                                                    magick.dir), pattern = "^convert\\.exe$", full.names = TRUE,
                                                          recursive = TRUE))) {
                    convert = shQuote(normalizePath(magick.path[1]))
                    message("but I can find it from the 'Program Files' directory: ",
                            magick.path)
                }
                else if (!inherits(try({
                    magick.path = utils::readRegistry("LyX.Document\\Shell\\open\\command", "HCR")
                }, silent = TRUE), "try-error")) {
                    convert = file.path(dirname(gsub("(^\"|\" \"%1\"$)", "", magick.path[[1]])), c("..", "../etc"), "imagemagick", "convert.exe")
                    convert = convert[file.exists(convert)]
                    if (length(convert)) {
                        convert = shQuote(normalizePath(convert))
                        message("but I can find it from the LyX installation: ", dirname(convert))
                    } else {
                        warning("No way to find ImageMagick!")
                        return()
                    }
                } else {
                    warning("ImageMagick not installed yet!")
                    return()
                }
                ## write it into ani.options() to save future efforts
                ani.options(convert = convert)
            }
            else {
                warning("Please install ImageMagick first or put its bin path into the system PATH variable")
                return()
            }
        }
    } else {
        ## GraphicsMagick
        version = ''
        if (!is.null(ani.options('convert'))) {
            version = try(cmd.fun(sprintf("%s -version", ani.options('convert')),
                              intern = TRUE, ignore.stdout = .ani.env$check, ignore.stderr = .ani.env$check))
        }
        if (!length(grep("GraphicsMagick", version))) {
            version = try(cmd.fun(sprintf("%s -version", convert), intern = TRUE,
                                  ignore.stdout = .ani.env$check, ignore.stderr = .ani.env$check))
            if (!length(grep("GraphicsMagick", version))) {
                warning("I cannot find GraphicsMagick with convert = ", shQuote(convert),
                        "; you may have to put the path of GraphicsMagick in the PATH variable.")
                return()
            }
        } else convert = ani.options('convert')
    }

    loop = ifelse(isTRUE(ani.options('loop')), 0, ani.options('loop'))
    convert = sprintf("%s -loop %s %s %s %s", convert, loop, extra.opts,
                      paste('-delay', interval * 100,
                            if (length(interval) == 1)
                            paste(files, collapse = ' ') else files,
                            collapse = ' '), shQuote(output.path))
    message("Executing: ", strwrap(convert, exdent = 4, prefix = '\n'))
    if (interactive()) flush.console()
    cmd = cmd.fun(convert, ignore.stdout = .ani.env$check, ignore.stderr = .ani.env$check)
    ## if fails on Windows using shell(), try system() instead of shell()
    if (cmd == 0 && .Platform$OS.type == "windows" && identical(cmd.fun, shell)) {
        cmd = system(convert, ignore.stdout = .ani.env$check, ignore.stderr = .ani.env$check)
    }
    if (cmd == 0) {
        message("Output at: ", output.path)
        if (clean)
            unlink(files)
        if (ani.options('autobrowse')) {
            if (.Platform$OS.type == 'windows')
                try(shell.exec(output.path)) else if (Sys.info()["sysname"] == "Darwin")
                    try(system(paste('open ', shQuote(output.path))), TRUE) else
            try(system(paste('xdg-open ', shQuote(output.path))), TRUE)
        }
        return(invisible(output.path))
    }
    else {
        message("There seems to be an error in the conversion...")
    }
}

##' A wrapper for the `gm convert' utility of GraphicsMagick.
##'
##' the function \code{gm.convert} is a wrapper for the command
##' \command{gm convert} of GraphicsMagick;
##' @rdname convert
##' @param ... arguments to be passed to \code{\link{im.convert}}
gm.convert = function(..., convert = "gm convert") {
    im.convert(..., convert = convert)
}
