% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/02-normalization.R
\name{normalize_logpost}
\alias{normalize_logpost}
\title{Normalize the joint posterior using AGHQ}
\usage{
normalize_logpost(
  optresults,
  k,
  whichfirst = 1,
  basegrid = NULL,
  ndConstruction = "product",
  ...
)
}
\arguments{
\item{optresults}{The results of calling \code{aghq::optimize_theta()}: see return value of that function.}

\item{k}{Integer, the number of quadrature points to use. I suggest at least 3. k = 1 corresponds to a Laplace
approximation.}

\item{whichfirst}{Integer between 1 and the dimension of the parameter space, default 1.
The user shouldn't have to worry about this: it's used internally to re-order the parameter vector
before doing the quadrature, which is useful when calculating marginal posteriors.}

\item{basegrid}{Optional. Provide an object of class \code{NIGrid} from the \code{mvQuad}
package, representing the base quadrature rule that will be adapted. This is only
for users who want more complete control over the quadrature, and is not necessary
if you are fine with the default option which basically corresponds to
\code{mvQuad::createNIGrid(length(theta),'GHe',k,'product')}.}

\item{ndConstruction}{Create a multivariate grid using a product or sparse construction?
Passed directly to \code{mvQuad::createNIGrid()}, see that function for further details. Note
that the use of sparse grids within \code{aghq} is currently \strong{experimental} and not supported
by tests. In particular, calculation of marginal posteriors is known to fail currently.}

\item{...}{Additional arguments to be passed to \code{optresults$ff}, see \code{?optimize_theta}.}
}
\value{
If k > 1, a list with elements:
\itemize{
\item{\code{nodesandweights}: }{a dataframe containing the nodes and weights for the adaptive quadrature rule, with the un-normalized and normalized log posterior evaluated at the nodes.}
\item{\code{thegrid}: }{a \code{NIGrid} object from the \code{mvQuad} package, see \code{?mvQuad::createNIGrid}.}
\item{\code{lognormconst}: }{the actual result of the quadrature: the log of the normalizing constant of the posterior.}
}

If k = 1, then the method returns
a numeric value representing the log of the normalizing constant computed using
a Laplace approximation.
}
\description{
This function takes in the optimization results from \code{aghq::optimize_theta()}
and returns a list with the quadrature points, weights, and normalization
information. Like \code{aghq::optimize_theta()}, this is designed for use only within
\code{aghq::aghq}, but is exported for debugging and documented in case you want to
modify it somehow, or something.
}
\examples{
# Same setup as optimize_theta
logfteta <- function(eta,y) {
  sum(y) * eta - (length(y) + 1) * exp(eta) - sum(lgamma(y+1)) + eta
}
set.seed(84343124)
y <- rpois(10,5) # Mode should be sum(y) / (10 + 1)
truemode <- log((sum(y) + 1)/(length(y) + 1))
objfunc <- function(x) logfteta(x,y)
funlist <- list(
  fn = objfunc,
  gr = function(x) numDeriv::grad(objfunc,x),
  he = function(x) numDeriv::hessian(objfunc,x)
)
opt_sparsetrust <- optimize_theta(funlist,1.5)
opt_trust <- optimize_theta(funlist,1.5,control = default_control(method = "trust"))
opt_bfgs <- optimize_theta(funlist,1.5,control = default_control(method = "BFGS"))

# Quadrature with 3, 5, and 7 points using sparse trust region optimization:
norm_sparse_3 <- normalize_logpost(opt_sparsetrust,3,1)
norm_sparse_5 <- normalize_logpost(opt_sparsetrust,5,1)
norm_sparse_7 <- normalize_logpost(opt_sparsetrust,7,1)

# Quadrature with 3, 5, and 7 points using dense trust region optimization:
norm_trust_3 <- normalize_logpost(opt_trust,3,1)
norm_trust_5 <- normalize_logpost(opt_trust,5,1)
norm_trust_7 <- normalize_logpost(opt_trust,7,1)

# Quadrature with 3, 5, and 7 points using BFGS optimization:
norm_bfgs_3 <- normalize_logpost(opt_bfgs,3,1)
norm_bfgs_5 <- normalize_logpost(opt_bfgs,5,1)
norm_bfgs_7 <- normalize_logpost(opt_bfgs,7,1)

}
\seealso{
Other quadrature: 
\code{\link{aghq}()},
\code{\link{get_log_normconst}()},
\code{\link{laplace_approximation}()},
\code{\link{marginal_laplace_tmb}()},
\code{\link{marginal_laplace}()},
\code{\link{optimize_theta}()},
\code{\link{plot.aghq}()},
\code{\link{print.aghqsummary}()},
\code{\link{print.aghq}()},
\code{\link{print.laplacesummary}()},
\code{\link{print.laplace}()},
\code{\link{print.marginallaplacesummary}()},
\code{\link{summary.aghq}()},
\code{\link{summary.laplace}()},
\code{\link{summary.marginallaplace}()}
}
\concept{quadrature}
