% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/abesspca.R
\name{abesspca}
\alias{abesspca}
\title{Adaptive best subset selection for principal component analysis}
\usage{
abesspca(
  x,
  type = c("predictor", "gram"),
  sparse.type = c("fpc", "kpc"),
  cor = FALSE,
  support.size = NULL,
  c.max = NULL,
  lambda = 0,
  always.include = NULL,
  group.index = NULL,
  splicing.type = 1,
  max.splicing.iter = 20,
  warm.start = TRUE,
  ...
)
}
\arguments{
\item{x}{A matrix object. It can be either a predictor matrix
where each row is an observation and each column is a predictor or
a sample covariance/correlation matrix.
If \code{x} is a predictor matrix, it can be in sparse matrix format
(inherit from class \code{"dgCMatrix"} in package \code{Matrix}).}

\item{type}{If \code{type = "predictor"}, \code{x} is considered as the predictor matrix.
If \code{type = "gram"}, \code{x} is considered as a sample covariance or correlation matrix.}

\item{sparse.type}{If \code{sparse.type = "fpc"}, then best subset selection performs on the first principal component;
If \code{sparse.type = "kpc"}, then best subset selection performs on the first \eqn{K} principal components.}

\item{cor}{A logical value. If \code{cor = TRUE}, perform PCA on the correlation matrix;
otherwise, the covariance matrix.
This option is available only if \code{type = "predictor"}.
Default: \code{cor = FALSE}.}

\item{support.size}{An integer vector. It represents the alternative support sizes when \code{sparse.type = "fpc"},
while each support size controls the sparsity of a principal component when \code{sparse.type = "kpc"}.
When \code{sparse.type = "fpc"} but \code{support.size} is not supplied,
it is set as \code{support.size = 1:min(ncol(x), 100)} if \code{group.index = NULL};
otherwise, \code{support.size = 1:min(length(unique(group.index)), 100)}.
When \code{sparse.type = "kpc"} but \code{support.size} is not supplied,
then for 20\% principal components,
it is set as \code{min(ncol(x), 100)} if \code{group.index = NULL};
otherwise, \code{min(length(unique(group.index)), 100)}.}

\item{c.max}{an integer splicing size. The default of \code{c.max} is the maximum of 2 and \code{max(support.size) / 2}.}

\item{lambda}{A single lambda value for regularized best subset selection. Default is 0.}

\item{always.include}{An integer vector containing the indexes of variables that should always be included in the model.}

\item{group.index}{A vector of integers indicating the which group each variable is in.
For variables in the same group, they should be located in adjacent columns of \code{x}
and their corresponding index in \code{group.index} should be the same.
Denote the first group as \code{1}, the second \code{2}, etc.
If you do not fit a model with a group structure,
please set \code{group.index = NULL} (the default).}

\item{splicing.type}{Optional type for splicing.
If \code{splicing.type = 1}, the number of variables to be spliced is
\code{c.max}, ..., \code{1}; if \code{splicing.type = 2},
the number of variables to be spliced is \code{c.max}, \code{c.max/2}, ..., \code{1}.
Default: \code{splicing.type = 1}.}

\item{max.splicing.iter}{The maximum number of performing splicing algorithm.
In most of the case, only a few times of splicing iteration can guarantee the convergence.
Default is \code{max.splicing.iter = 20}.}

\item{warm.start}{Whether to use the last solution as a warm start. Default is \code{warm.start = TRUE}.}

\item{...}{further arguments to be passed to or from methods.}
}
\value{
A S3 \code{abesspca} class object, which is a \code{list} with the following components:
\item{coef}{A \eqn{p}-by-\code{length(support.size)} loading matrix of sparse principal components (PC),
where each row is a variable and each column is a support size;}
\item{nvars}{The number of variables.}
\item{sparse.type}{The same as input.}
\item{support.size}{The actual support.size values used. Note that it is not necessary the same as the input if the later have non-integer values or duplicated values.}
\item{ev}{A vector with size \code{length(support.size)}. It records the explained variance at each support size.}
\item{pev}{A vector with the same length as \code{ev}. It records the percentage of explained variance at each support size.}
\item{var.all}{If \code{sparse.type = "fpc"},
it is the total variance of the explained by first principal component;
otherwise, the total standard deviations of all principal components.}
\item{call}{The original call to \code{abess}.}
}
\description{
Adaptive best subset selection for principal component analysis
}
\details{
Adaptive best subset selection for principal component analysis aim
to solve the non-convex optimization problem:
\deqn{\arg\max_{v} v^\top \Sigma v, s.t.\quad v^\top v=1, \|v\|_0 \leq s, }
where \eqn{s} is support size. A generic splicing technique is implemented to
solve this problem.
By exploiting the warm-start initialization, the non-convex optimization
problem at different support size (specified by \code{support.size})
can be efficiently solved.
}
\examples{
\donttest{
library(abess)

## predictor matrix input:
head(USArrests)
pca_fit <- abesspca(USArrests)
pca_fit

## covariance matrix input:
pca_fit <- abesspca(stats::cov(USArrests), type = "gram")
pca_fit

## robust covariance matrix input:
rob_cov <- MASS::cov.rob(USArrests)[["cov"]]
rob_cov <- (rob_cov + t(rob_cov)) / 2
pca_fit <- abesspca(rob_cov, type = "gram")
pca_fit

## K-component principal component analysis
pca_fit <- abesspca(USArrests,
  sparse.type = "kpc",
  support.size = c(1, 2)
)
coef(pca_fit)
}
}
\references{
A polynomial algorithm for best-subset selection problem. Junxian Zhu, Canhong Wen, Jin Zhu, Heping Zhang, Xueqin Wang. Proceedings of the National Academy of Sciences Dec 2020, 117 (52) 33117-33123; DOI: 10.1073/pnas.2014241117

Sparse principal component analysis. Hui Zou, Hastie Trevor, and Tibshirani Robert. Journal of computational and graphical statistics 15.2 (2006): 265-286.
}
\seealso{
\code{\link{print.abesspca}},
\code{\link{coef.abesspca}},
}
\author{
Jin Zhu, Junxian Zhu, Ruihuang Liu, Junhao Huang, Xueqin Wang
}
