#' check_TemporalGSSA()
#'
#' This function of the package 'TemporalGSSA' checks whether the user-defined
#' data is suitable for further processing.
#'
#' The number of observations must be at least 30 (consective runs of 'GillespieSSA') 
#' to generate data that is statistically significant. The user must also enter a choice that
#' specifies the statistical modality utilized to compute a representative timestep
#'  (1, mean; 2, median; 3, random; 4, all). Both arguments are mandatory.
#'
#' @param data Formatted data from the output of a GillespieSSA run
#' This is a mandatory argument.
#
#' @param type A mandatory integer argument that indicates the statistic to
#' be used to compute the representative timestep for a set of observations.
#' Here: "1" is the MEAN, "2" is the MEDIAN, "3" is a random choice, "4" ALL
#'
#' @return flag A numeric indicator of the suitability of the user-defined input
#' data for further computations.

check_TemporalGSSA <- function(data,type)
{
       flag <- 0
       nro <- nrow(data.matrix(data))
       if((nro > 0) && (nro < 30)) 
                    {
                     print("The number of rows for your data are less than 30. The data generated will not be significant. Please, revise this")
                     if(type == "NULL"){print("Please enter an integer from 1 and 4. These indicate the type of parameterization to be done to the timestep (1, MEAN; 2, MEDIAN; 3, RANDOM; 4, ALL)");flag <- 1}
                     else{flag <- 1}                           
                     }
        else if(nro >= 30)
                    {
                     if(type == "NULL"){print("This argument cannot be left blank. Please enter an integer from 1 and 4. These indicate the type of parameterization to be done to the timestep (1, MEAN; 2, MEDIAN; 3, RANDOM; 4, ALL)");flag <- 1;}
                     else{flag <- 0}
                     }
        else if( (data == "NULL") && (type == "NULL") ) 
                    {
                     print("You have not entered any data. Please, revise this.")
                     print("Please indicate your choice of parameterization for the timestep (1, MEAN; 2, MEDIAN; 3, RANDOM; 4, ALL)")
                     flag <- 1
                     }


return(flag)
}

#' calculate_TemporalGSSA()
#
#' This is the main function of the package 'TemporalGSSA'.
#'
#' 'TemporalGSSA', may be considered a wrapper for the R-package 'GillespieSSA'.
#' The formatted data generated by consecutive runs of 'GillespieSSA' for molecules of a generic 
#'  biochemical network will generate a dataset of timesteps along with changes to the 
#'  corresponding number of molecules.
#'
#' The temporal profile of a molecule is a pre-requisite to comprehend its' behaviour in vivo.
#'
#' This is accomplished by selected a representative timestep for a set of observations
#' or consecutive runs (n >= 30). A linear model of the numbers of each molecule is
#' created with the assocated timestep from these observations. The coefficients of
#' this model (slope, constant) are then incorporated into a second linear regression model.
#'
#' Here, the independent variable is the representative timestep. The generated data, i.e.,
#' the imputed molecule number for an in-silico experiment with n >=30 observations.
#'
#' This can be replicated with one or more sets of observations or runs. These "technical replicates"
#' can be averaged and will constitute the time-dependent data point of each molecule for a particular
#' simulation time. For varying simulation times these data will generate time-dependent trajectories 
#' for each molecule of the biochemical network under study.
#'
#'
#' @param data Formatted data from the output of consecutive 'GillespieSSA' runs.
#' This is a mandatory argument.
#
#' @param type A mandatory integer argument that indicates the statistic that will be used
#' to compute the representative timestep for a set of observations.
#' Here: "1" is the MEAN, "2" is the MEDIAN, "3" is a random choice and "4" is ALL
#'
#' @param out An optional logical argument that control the console output with default being "TRUE"
#' Here, TRUE curtails the output while FALSE allows the verbose output of all the coefficents .
#'
#' @return code A numerically encoded ('0', success; '1', no success) text message to the
#' user indicating the outcome of "TemporalGSSA".
#'
#' The examples listed below can be run directly from the console as example("calculate_TemporalGSSA") 
#'
#' @examples
#' \dontshow{
#' valid <- matrix(c(4.99592056104662E-19,499995,500003,52076,856074,408144,408147,213587,305440,305446,305445,1039792,9.0895286250952E-19,499992,500000,29946,842290,372229,372231,4384,132154,132160,132167,1097848,5.66871925230001E-19,499989,500008,43979,856199,400171,400175,177259,277086,277096,277095,1055869,9.13431925017368E-19,499997,499999,30174,842560,372728,372730,22234,128214,128213,128214,1097112,6.66856629255775E-19,499997,500007,36194,853340,389533,389529,124479,234948,234950,234950,1074288,9.18748453328027E-19,499998,500001,29951,841897,371842,371845,466464,128159,128156,128159,1098219,6.33923848260308E-19,499996,499997,38414,854111,392517,392523,141620,249102,249102,249106,1069085,6.61025475380139E-19,499995,500004,36842,852757,389594,389595,127161,237566,237572,237572,1073574,6.46404578600321E-19,499997,499993,37546,853998,391541,391543,134830,243289,243287,243290,1070928,8.71251875548091E-19,499995,500002,30238,844659,374894,374894,23273,148382,148384,148386,1094886,6.42456762267738E-19,499994,500001,37659,854377,392030,392032,137247,245215,245219,245221,1070325,6.50000982450828E-19,499997,500002,37403,853794,391193,391195,132931,241735,241744,241741,1071411,6.51703301709049E-19,499996,500000,37294,853141,390431,390434,132095,241663,241666,241664,1072285,9.19733238113478E-19,499997,499998,29641,842522,372160,372162,74747,128165,128167,128168,1098208,8.8945111346424E-19,499999,499998,29956,844208,374158,374163,14760,140602,140598,140602,1095898,6.55496374724788E-19,499994,499999,37078,853792,390867,390867,129538,238669,238673,238673,1072070,6.40478506078292E-19,499997,499993,38083,854148,392221,392220,138163,245947,245945,245954,1069723,8.46044969800135E-19,499996,500004,30517,845908,376418,376424,34899,158483,158480,158477,1093077,9.19232350704206E-19,499994,500002,29541,842618,372151,372156,64364,127850,127852,127852,1098330,9.05290488116942E-19,499999,500002,30116,843048,373157,373161,858885,133256,133260,133258,1096737,9.0739154712218E-19,499990,500005,29854,843198,373043,373049,4709,131663,131673,131675,1097125,9.18051289626637E-19,499995,499999,29866,841910,371768,371769,63232,128229,128235,128241,1098386,9.18014676040227E-19,499996,500000,29956,842427,372378,372384,37733,127620,127621,127618,1097675,8.99677088174691E-19,499993,500003,29947,842729,372672,372673,8741,136069,136075,136076,1097396,9.08400495196285E-19,499992,500009,29926,843132,373051,373055,4843,131792,131799,131796,1097043,9.17671540274208E-19,499996,499999,29855,842155,372001,372007,97878,128919,128919,128921,1098164,9.15629173860131E-19,499995,500001,30161,842024,372181,372183,255,128075,128079,128081,1097681,9.18787494819808E-19,500000,500003,30059,842445,372498,372502,758505,127500,127502,127503,1097454,9.19961013729775E-19,499999,500007,29565,842688,372249,372251,657849,127746,127748,127745,1098192,9.17770058854612E-19,499998,500000,29852,841831,371681,371686,5363,128318,128316,128317,1098480),byrow=TRUE,nrow=30)
#' }
#' \dontshow{
#' nvalid <- matrix(c(4.99592056104662E-19,499995,500003,52076,856074,408144,408147,213587,305440,305446,305445,1039792,9.0895286250952E-19,499992,500000,29946,842290,372229,372231,4384,132154,132160,132167,1097848,5.66871925230001E-19,499989,500008,43979,856199,400171,400175,177259,277086,277096,277095,1055869,9.13431925017368E-19,499997,499999,30174,842560,372728,372730,22234,128214,128213,128214,1097112),byrow=TRUE,nrow=4)
#' }
#' calculate_TemporalGSSA(valid,1)
#' calculate_TemporalGSSA(valid,4)
#' calculate_TemporalGSSA(nvalid,2)
#'
#'  
#' @import stats
#' @export

calculate_TemporalGSSA <- function(data,type,out)
{

if( (missing(data)) || (missing(type))  ){data <- "NULL"; type  <- "NULL"}
if(missing(out)){out <- "TRUE"}
code <- check_TemporalGSSA(data,type)
p <- array(dim = c(1,ncol(data)))
w <- array(dim = c(nrow(data),1))
p4 <- array(dim = c(3,ncol(data)+1))
su <- list(ncol(data))
if(code == 0)
        {
                  print("Input data checked......[OK]")
                  print("Commencing data transformation...")
                  r <- y <- s <- so <- 0
                  r1 <- r2 <- r3 <- so1 <- so2 <- so3 <- 0
                  index <- r_title <- ""

                  w <- data[,1]
                        if(type == 1){r <- mean(w);p[1,1] <- r;r_title = "as a function of the MEAN predictor of the representative timestep"}
                       else if(type == 2){r <- median(w);p[1,1] <- r;r_title = "as a function of the MEDIAN predictor of the representative timestep"}
                       else if(type == 3){r <- sample(w,1);p[1,1] <- r;r_title = "as a function of RANDOM predictor of the representative timestep"}
                       else if(type == 4){r1 <- mean(w);r2 <- median(w);r3 <- sample(w,1);p4[1,1] <- "MEAN";p4[2,1] <- "MEDIAN";p4[3,1] <- "RANDOM";
                                                     p4[1,2] <- r1;p4[2,2] <- r2;p4[3,2] <- r3;r_title = "as functions of all predictors of the representative timestep"}

                  if(type < 4)
                               {
                                 for(n in 2:ncol(data))
                                       {
                                          y <- lm(data[,n]~w);id <- paste("Data for metabolite",n-1,sep="_")
                                          s <- matrix(coefficients(y))
                                          so <- round((s[2,1]*r) + s[1,1],0)
                                          p[1,n] <- so;su[n-1] <- list(summary(y))
                                       }
                                print(index <- paste("OUTPUT SUMMARY FOR (n = ",n-1,") metabolites (",r_title,")",sep=""))
                                print("The details of these calculations are available by typing output$a2")  
                                output <- list(a1=p,a2=su)
                                if(out == "FALSE"){print(output$a2)}
                                print(output$a1)
                               }
                   else
                             {
                                for(n in 2:ncol(data))
                                      {
                                         y <- lm(data[,n]~w);id <- paste("Data for metabolite",n-1,sep="_")
                                         s <- matrix(coefficients(y))
                                         so1 <- round((s[2,1]*r1) + s[1,1],0);so2 <- round((s[2,1]*r2) + s[1,1],0);so3 <- round((s[2,1]*r3) + s[1,1],0);
                                         p4[1,n+1] <- so1;p4[2,n+1] <- so2;p4[3,n+1] <- so3
                                         su[n-1] <- list(summary(y))
                                      }
                                print(index <- paste("OUTPUT SUMMARY FOR (n = ",n-1,") metabolites (",r_title,")",sep=""))
                                print("The details of these calculations are available by typing output$a2")  
                                output <- list(a1=p4,a2=su)
                                if(out == "FALSE"){print(output$a2)}
                                 print(output$a1)
                             }
                    
               }
else
           {
            print("Data transformation.....[FAIL]")
            print("Please revise your input")
           }

return(code)
}
