% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RNV.for.simul.R
\name{RNV.for.simul}
\alias{RNV.for.simul}
\title{Compute RNV for simulation}
\usage{
RNV.for.simul(res_sim,n_fun,N_fun,correl_fun,beta_fun=NULL,
end.mean=TRUE)
}
\arguments{
\item{res_sim}{Dataframe corresponding to result of one simulation. \emph{See details}.}

\item{n_fun}{Integer number indicating the number of enzymes.}

\item{N_fun}{Numeric. Population size}

\item{correl_fun}{Character string indicating the abbreviation of the constraint applied on the system}

\item{beta_fun}{Matrix of co-regulation coefficients}

\item{end.mean}{Logical. If \code{FALSE}, compute RNV size mean for all rows of \code{res_sim}. If \code{TRUE}, compute RNV size mean for last half of \code{res_sim} rows.}
}
\value{
Invisible list of 7 elements:
\itemize{
   \item \code{$RNV_delta}: list of \code{n} elements, one by enzyme.
   Each element contains a numeric matrix of \code{nb_resid} rows and two or four columns.
   For each enzyme \code{i} (a list element), each row corresponds to one resident and columns are actual mutation size \eqn{\delta_i} corresponding respectively to inferior bound (col 1) and superior bound (col 2) of RNV (x2 if there is a second RNV).
   Each row is in fact a result of \code{\link{RNV.delta.all.enz}}.
   \item \code{$RNV_enz}: same structure as \code{$RNV_delta}, but for mutant enzyme concentrations at RNV limits, \emph{i.e.} \eqn{E_i + \delta_i} for each target enzyme \code{i}.
   \item \code{$RNV_size}: list of one or two elements (depending RNV on RNV number).
   Each element contains a matrix of \code{n} columns (one by enzyme) and \code{nb_resid} rows.
   Each cell is the RNV size \emph{(see details)}.
   \item \code{$RNV_size_divEtot}: same structure as \code{$RNV_size}, but for RNV size divided by total concentration of corresponding resident.
   \item \code{$RNV_proxy}: numeric matrix of one or two rows (depending on RNV number) and \code{n} columns.
   Each cell is the mean of RNV size \emph{(see details)}.
   \item \code{$RNV_proxy_divEtot}: same structure as \code{$RNV_proxy}, and contains mean of RNV size divided by total concentration.
   \item \code{$RNV_flux}: numeric matrix of two columns (inferior and superior limits of neutral zone) and \code{nb_resid} rows.
   Each cell is the flux value at neutral zone limits.
   \item \code{$nb_RNV}: number of RNV. If constraint is \code{"SC"} or \code{"RegPos"}, there is one RNV, else two.
   \item \code{$limits_NZ}: numeric vector of the two limits of neutral zone
}



Note that \code{n} is the number of enzymes. \code{nb_resid} is the number of resident and is also the rows number of \code{res_sim}.
}
\description{
Computes different elements at RNV for evolution simulation
}
\details{
This function is designed to computes RNV of \emph{one} simulation launched by \code{\link{simul.evol.enz.one}}.
Input \code{res_sim} is a result of \code{\link{simul.evol.enz.one}}.
If you used \code{\link{simul.evol.enz.multiple}}, input \code{tabR} for parameter \code{res_sim}.
If input is several simulations, remember that number of rows for one simulation is \code{npt}. 
\emph{See below.}
  
The \emph{Range of Neutral Variations} (RNV) are mutant concentration values such as coefficient selection is between \eqn{1/(2N)} and \eqn{-1/(2N)}.

Inferior (resp. superior) bound of RNV corresponds to selection coefficient equal to \eqn{-1/(2N)} (resp. \eqn{1/(2N)}).

Function \code{RNV.for.simul} computes the actual mutation effect \eqn{\delta_i} at RNV bounds (where \eqn{i} is the enzyme targeted by the mutation),
but also mutant concentration at RNV bounds and the RNV size.



Depending on applied constraint \code{correl_fun}, it exists 1 or 2 RNV.
In case of independence (\code{"SC"}) or positive regulation between all enzymes (\code{"RegPos"}), flux has no limit and there is only one RNV.
In other cases (competition and/or negative regulation), because flux can reach a maximum, there is two RNV: 
a "near" one, for small mutations, and a "far" one for big mutations that put mutants on the other side of flux dome.



\bold{RNV size}

The RNV size is the absolute value of \eqn{\delta_i^sup} minus \eqn{\delta_i^inf}.
If there is no superior bounds but there is two RNVs, RNV size is obtained by the difference of the two \eqn{\delta_i^inf}.
 
The mean of RNv size is the mean of every resident for each enzyme. If \code{end.mean=TRUE}, only last half of resident (the last half of \code{res_sim} rows) are used to compute RNV mean.
 
 
\bold{Use of \code{res_sim}}

\code{res_sim} is a numeric matrix of \code{(3*n+4)} columns and at least 2 rows.
Respective columns are: concentrations (\code{1:n}), kinetic parameters (\code{n+1:2n}), total concentration (\code{2n+1}), total kinetic (\code{2n+2}), flux/fitness (\code{2n+3}) and activities (\code{2n+4:3n+3}), corresponding to \emph{(E1 to En, kin_1 to kin_n, Etot, kin_tot, J, A1 to An)}.
\emph{See function \code{\link{simul.evol.enz.one}}.}

\code{res_sim} is normally output \code{$res_sim} of function \code{\link{simul.evol.enz.one}}.
Output \code{$tabR} of function \code{\link{simul.evol.enz.multiple}} is also possible, by selecting a simulation with \code{x$tabR[x$tabR$sim==i,]} where \code{i} is simulation number.
If input is several simulations, remember that number of rows for one simulation is \code{npt}. 

Other parameters (\code{n_fun, N_fun, correl_fun, beta_fun}) are available in output \code{$param} of simulation functions.
}
\examples{

 #### Construction of false simulation
#for 2 resident genotypes and 3 enzymes
n <- 3
Er <- c(30,30,30)
kin <- c(1,10,30)
Keq <- c(1,1,1)
A <- activities(kin,Keq)
beta <- matrix(c(1,10,5,0.1,1,0.5,0.2,2,1),nrow=3)
B <- compute.B.from.beta(beta)
correl <- "RegPos"
N <- 1000

#on one line
first_res <- cbind(t(Er),t(kin),sum(Er),sum(kin),flux(Er,A),t(A))

#second resident = theoretical equilibrium of first resident
Eq_th <- 100*predict_th(A,correl,B)$pred_e
second_res <- cbind(t(Eq_th),t(kin),sum(Eq_th),sum(kin),flux(Eq_th,A),t(A))

false_sim <- rbind(first_res,second_res)

RNV_elements <- RNV.for.simul(false_sim,n,N,correl,beta)

RNV_elements$RNV_delta #apparent mutation size at RNV for enzymes 1, 2 and 3
RNV_elements$RNV_enz #concentrations
RNV_elements$RNV_size #RNV size
RNV_elements$RNV_proxy #RNV size mean
RNV_elements$RNV_flux #flux at neutral zone

\donttest{
#With saved simulation
data(data_sim_RegPos)
RNV_elements <- RNV.for.simul(data_sim_RegPos$tabR,data_sim_RegPos$param$n,
data_sim_RegPos$param$N,data_sim_RegPos$param$correl,data_sim_RegPos$param$beta)
}


}
\seealso{
See function \code{\link{RNV.delta.all.enz}} to see how \eqn{\delta} is computed. 

Use function \code{\link{simul.evol.enz.one}} to launch a simulation, or \code{\link{simul.evol.enz.multiple}} for several simulations.
}
