\name{renewalTestPlot}
\alias{renewalTestPlot}

\title{Non-Parametric Tests for Renewal Processes}
\description{
  Performs and displays rank based tests checking if a spike train is a
  renewal process
}
\usage{
renewalTestPlot(spikeTrain, lag.max = NULL,
                d=max(c(2,sqrt(length(spikeTrain)) \%/\% 5)),
                orderPlotPch=ifelse(length(spikeTrain)<=600,1,"."),
                ...)
}

\arguments{
  \item{spikeTrain}{a \code{spikeTrain} object or a vector which can be
    coerced to such an object.}
  \item{lag.max}{argument passed to \code{\link{acf.spikeTrain}}.}
  \item{d}{an integer >= 2, the number of divisions used for the Chi 2
    test. The default value is such that under the null hypothesis at
    least 25 events should fall in each division.}
  \item{orderPlotPch}{\code{pch} argument for the order plots.}
  \item{\dots}{additional arguments passed to function \code{\link{chisq.test}}.}
}
\details{
  \code{renewalTestPlot} generates a 4 panel plot. The 2 graphs making
  the top row are qualitative and display the rank of inter-spike
  interval (ISI) k+1 versus the rank of ISI k (left graph) and the rank of ISI k+2
  versus the one of ISI k (right graph). 
  The bottom left graph displays the autocorrelation function of the ISIs
  and is generated by a call to \code{\link{acf.spikeTrain}}. The bottom right
  graph display the result of a Chi square test performed on the ranks
  at different lags. More precisely, for each considered lag j (from 1 to
  \code{lag.max}) the square within which the rank of ISI k+1 vs the one
  of ISI k is found is splited in \eqn{d^2}{d^2} cells. This decomposition
  into cells is shown on the two graphs of the top row. Under the
  renewal process hypothesis the points should be uniformly distributed
  with a density \eqn{\frac{N}{d^2}}{N/d^2}, where N is the number of
  ISIs. The sum other rows and other columns is moreover exactly
  \eqn{\frac{N}{d}}{N/d}. The upper graphs are therefore graphical
  displays of two-dimensional contingency tables. A chi square test for
  two-dimensional contingency tables (function \code{\link{chisq.test}})
  is performed on the table generated at each lag j. The resulting Chi 2
  value is displayed vs the lag. The 95\% confidence region appears as a
  clear grey rectangle, the value falling within this region appear as
  black dots and the ones falling out appear as dark grey triangles. 
}
\value{
  Nothing is returned, the function is used for its side effect: a plot
  is generated.
}
\author{Christophe Pouzat  \email{christophe.pouzat@gmail.com} }
\note{
  You should not use a too large value for \code{d} otherwise the Chi 2
  values will be too approximative and warnings will be printed.
  If your process is a renewal process you should have on average 5\% of
  the points on the bottom right graph appearing as dark triangles.
}
\seealso{
  \code{\link{acf}},
  \code{\link{varianceTime}},
  \code{\link{acf.spikeTrain}}
}
\examples{
## Apply the test of Ogata (1988) shallow shock data
data(ShallowShocks)
renewalTestPlot(ShallowShocks$Date,d=3)

## Apply the test to the second and third neurons of the cockroachAlSpont
## data set
## load spontaneous data of 4 putative projection neurons
## simultaneously recorded from the cockroach (Periplaneta
## americana) antennal lobe
data(CAL1S)
## convert data into spikeTrain objects
CAL1S <- lapply(CAL1S,as.spikeTrain)
## look at the individual trains
## first the "raw" data
CAL1S[["neuron 1"]]
## next some summary information
summary(CAL1S[["neuron 1"]])
## next the renewal tests
renewalTestPlot(CAL1S[["neuron 1"]])

## Simulate a renewal log normal train with 500 isi
isi.nb <- 500
train1 <- c(cumsum(rlnorm(isi.nb+1,log(0.01),0.25)))
## make the test
renewalTestPlot(train1)

## Simulate a (non renewal) 2 states train
myTransition <- matrix(c(0.9,0.1,0.1,0.9),2,2,byrow=TRUE)
states2 <- numeric(isi.nb+1) + 1
for (i in 1:isi.nb) states2[i+1] <- rbinom(1,1,prob=1-myTransition[states2[i],])+1
myLnormPara2 <- matrix(c(log(0.01),0.25,log(0.05),0.25),2,2,byrow=TRUE)
train2 <-
cumsum(rlnorm(isi.nb+1,myLnormPara2[states2,1],myLnormPara2[states2,2]))
## make the test
renewalTestPlot(train2)
}
\keyword{ts}
\keyword{survival}
