#' @title Read in Raven Hydrograph file
#'
#' @description
#' rvn_hyd_dygraph plots modeled vs observed hydrographs when supplied with hydrograph data
#' structure read using \code{\link{rvn_hyd_read}}
#'
#' @param hy hydrograph data structure generated by \code{\link{rvn_hyd_read}} routine
#' @param timezone data timezone; defaults to UTC
#' @param basins list of subbasin names from hydrograph file. Each subbasin creates separate dygraph plots
#' @param figheight height of figure, in pixels
#'
#' @return {res}{ a list of plot handles to dygraph plots}
#'
#' @examples
#'
#' # read in RavenR sample hydrographs data
#' hy <- rvn_hydrograph_data
#'
#' # view contents for subbasin 36 as dyGraph
#' dyplots <- rvn_hyd_dygraph(hy,basins="Sub36")
#' dyplots
#'
#' # view contents for all basins in hydrograph data
#' rvn_hyd_dygraph(hy)
#'
#' @export rvn_hyd_dygraph
#' @importFrom purrr map
#' @importFrom dygraphs dygraph dyRangeSelector
rvn_hyd_dygraph <- function(hy, timezone="UTC",basins="", figheight=400) {

  if (basins==""){
    basins<-sub('_obs', "", colnames(hy$hyd))
    basins<-sub('_res', "", basins)
    basins<-unique(basins)
    basins<-basins[-1] # no precip ts
  }

  plotfunc <- function(basin){
    hyd <- rvn_hyd_extract(subs = basin, hyd = hy)
    flow <- merge(hyd$sim, hyd$obs)
    return(dygraph(flow, group = 'flows', main = basin, height=figheight) %>%
             dyRangeSelector())
  }

  dyplots <- purrr::map(basins, plotfunc) # calls plotfunc for each basin in list basins
  return(dyplots)
}
