#----------------------------------------------------------------------------
# RSuite
# Copyright (c) 2017, WLOG Solutions
#
# Utilities for retrieving project package, master scripts and all further
# dependencies.
#----------------------------------------------------------------------------

#'
#' Detects direct uninstalled project dependencies
#'
#' @param params object of rsuite_project_params class
#'
#' @return object of versions class containing direct project dependencies
#'   which are not installed in project local environment.
#'
#' @keywords internal
#' @noRd
#'
collect_uninstalled_direct_deps <- function(params) {
  dep_vers <- collect_prj_direct_deps(params)
  installed <- as.data.frame(utils::installed.packages(params$lib_path),
                             stringsAsFactors = FALSE)[, c("Package", "Version")]
  dep_vers <- vers.rm_acceptable(dep_vers, installed)
}

#'
#' Detects all project (project packages and master scripts) direct dependencies.
#'
#' @param params object of rsuite_project_params class
#'
#' @return object of versions class containing all direct project dependencies
#'   together with their version requirements.
#'
#' @keywords internal
#' @noRd
#'
collect_prj_direct_deps <- function(params) {
  pkg_vers <- collect_pkgs_direct_deps(params)
  msc_vers <- collect_mscs_direct_deps(params)
  prj_vers <- vers.union(pkg_vers, msc_vers)
  return(prj_vers)
}

#'
#' Detects all project required support packages.
#'
#' @param params object of rsuite_project_params class
#' @param vanilla if TRUE detects only base supportive packages. (type: logical(1))
#'
#' @return object of versions class containing all support packages required.
#'   Version requirements for support packages included should be empty.
#'
#' @keywords internal
#' @noRd
#'
collect_prj_support_pkgs <- function(params, vanilla = FALSE) {
  prj_packages <- build_project_pkgslist(params$pkgs_path) # from 51_pkg_info.R

  requires_roxygen <- function(pkg_path) {
    has_rgen_mans <- lapply(X = list.files(file.path(pkg_path, "man"), ".+[.]Rd$", full.names = TRUE),
                            FUN = function(rd_file) {
                              if (!all(grepl("^% Generated by roxygen", readLines(rd_file, n = 1)))) {
                                return(FALSE)
                              }
                              return(TRUE)
                            })
    has_rgen_mans <- any(unlist(has_rgen_mans))

    ns_path <- file.path(pkg_path, "NAMESPACE")
    has_rgen_ns <- !file.exists(ns_path) || any(grepl("^# Generated by roxygen", trimws(readLines(ns_path))[1]))

    return(has_rgen_mans || has_rgen_ns)
  }

  sup_pkgs <- unlist(lapply(X = names(prj_packages),
                            FUN = function(pkg_dir) {
                              sup_pkgs <- "devtools"

                              pkg_path <- file.path(params$pkgs_path, pkg_dir)
                              desc_file <- file.path(pkg_path, "DESCRIPTION")
                              stopifnot(file.exists(desc_file))

                              desc <- read.dcf(desc_file)

                              if (requires_roxygen(pkg_path)) {
                                sup_pkgs <- c(sup_pkgs, "roxygen2")

                                if ("RoxygenExtraRoclets" %in% colnames(desc)) {
                                  roclets <- trimws(strsplit(desc[1, "RoxygenExtraRoclets"], ", ")[1])

                                  unspec_roclets <- roclets[!grepl("^[a-zA-Z]+::", roclets)]
                                  assert(length(unspec_roclets) == 0,
                                         "Some extra roclets in %s are underspecified: %s",
                                         pkg_dir, paste(unspec_roclets, collapse = ", "))

                                  roc_pkgs <- gsub("^([a-zA-Z]+)::.+$", "\\1", roclets)
                                  sup_pkgs <- c(sup_pkgs, roc_pkgs)
                                }
                              }

                              if (devtools::uses_testthat(pkg = pkg_path)) {
                                sup_pkgs <- c(sup_pkgs, "testthat")
                              }

                              if (any(vanilla)) {
                                return(sup_pkgs)
                              }

                              if ("VignetteBuilder" %in% colnames(desc)) {
                                sup_pkgs <- c(sup_pkgs, desc[1, "VignetteBuilder"])
                              }

                              tests_path <- file.path(pkg_path, "tests")
                              if (dir.exists(tests_path)) {
                                sup_pkgs <- c(sup_pkgs,
                                              collect_dir_script_deps(tests_path, recursive = FALSE))
                              }

                              return(sup_pkgs)
                            }))

  if (!any(vanilla)) {
    prj_tests_path <- file.path(params$prj_path, "tests")
    if (dir.exists(prj_tests_path)) {
      sup_pkgs <- c(sup_pkgs,
                    collect_dir_script_deps(prj_tests_path, recursive = FALSE))
    }
    if ("knitr" %in% sup_pkgs) {
      sup_pkgs <- c(sup_pkgs, "rmarkdown")
    }
  }

  support_vers <- vers.build(unique(sup_pkgs))

  # collect suggested packages
  suggest_vers <- do.call(
    "vers.union",
    lapply(X = names(prj_packages),
           FUN = function(pkg_dir) {
             sugs <- desc_retrieve_dependencies(params$pkgs_path, pkg_dir, # from 51_pkg_info.R
                                                fields = "Suggests")
             vers.from_deps(sugs, prj_packages[[pkg_dir]])
           }))

  suggest_vers <- vers.rm(suggest_vers,
                          setdiff(vers.get_names(suggest_vers), vers.get_names(support_vers)))

  return(vers.union(suggest_vers, support_vers))
}


#'
#' Looks for package DESCRIPTION files and retrieves direct dependencies from them.
#'
#' @param params object of rsuite_project_params class
#' @param prj_pkgs detect dependencies only for specified project packages. If
#'   NULL detect for all. (type: character, default: NULL)
#'
#' @return object of versions class containing all direct dependencies of project
#'   packages together with their version requirements.
#'
#' @keywords internal
#' @noRd
#'
collect_pkgs_direct_deps <- function(params) {
  prj_packages <- build_project_pkgslist(params$pkgs_path) # from 51_pkg_info.R
  pkgs_vers <- do.call("vers.union",
                       lapply(X = names(prj_packages),
                              FUN = function(pkg_dir) {
                                collect_single_pkg_direct_deps(params, pkg_dir, prj_packages[[pkg_dir]])
                              }))

  unfeasibles <- vers.get_unfeasibles(pkgs_vers)
  assert(length(unfeasibles) == 0,
         "Packages with unfeasible requirements detected: %s", paste(unfeasibles$pkg, collapse = ", "))

  # Check R version
  req_r_ver <- vers.get(pkgs_vers, "R")
  if (nrow(req_r_ver)) {
    cur_r_ver <- sprintf("%s.%s", R.version$major, R.version$minor)
    assert( (is.na(req_r_ver$vmin) || req_r_ver$vmin <= cur_r_ver)
            && (is.na(req_r_ver$vmax) || req_r_ver$vmax >= cur_r_ver),
            "R version(%s) does not meet requirements: it must be in range %s .. %s",
            cur_r_ver, req_r_ver$vmin, req_r_ver$vmax)
  }

  pkgs_vers <- vers.rm_base(pkgs_vers)
  return(pkgs_vers)
}


#'
#' Retrieve dependencies with requirements for a single project package
#'
#' @param params object of rsuite_project_params class
#' @param pkg_dir package folder (type: character)
#' @param pkg_name package name as declared in DESCRIPTION (type: character)
#'
#' @return object of versions class containing all direct dependencies of package
#'   together with their version requirements.
#'
#' @keywords internal
#' @noRd
#'
collect_single_pkg_direct_deps <- function(params, pkg_dir, pkg_name) {
  deps <- desc_retrieve_dependencies(params$pkgs_path, pkg_dir) # from 51_pkg_info.R
  vers.from_deps(deps, pkg_name)
}


#'
#' Looks for master scripts and retrieves all direct dependencies from them.
#'
#' @param params object of rsuite_project_params class
#'
#' @return object of versions class containing all direct dependencies of master
#'   scripts. Of cause master scrips cannot enforce version requirements so
#'   versions object does not contain requirements on package versions.
#'
#' @keywords internal
#' @noRd
#'
collect_mscs_direct_deps <- function(params) {
  pkgs <- collect_dir_script_deps(dir = params$script_path, recursive = TRUE)
  mscs_vers <- vers.build(unique(pkgs))
  mscs_vers <- vers.rm_base(mscs_vers)
  return(mscs_vers)
}

#'
#' Detects all dependencies in all script files in folder passed.
#'
#' @param dir folder to find scrips in. (type: character(1))
#' @param recursive if TRUE scripts will be detected recursively. (type: logical(1))
#'
#' @return vector of all libraries detected in scripts. (type: character(N))
#'
#' @keywords internal
#' @noRd
#'
collect_dir_script_deps <- function(dir, recursive = TRUE) {
  script_files <- list.files(path = dir, pattern = "*.(r|R|Rmd)$",
                             recursive = recursive, full.names = TRUE)
  pkgs <- unlist(lapply(
    X = script_files,
    FUN = function(sf){
      lns <- readLines(sf)
      loads <- lns[grepl("^\\s*(require|library)\\((.+)\\)", lns)]
      loads <- gsub("\\s+", "", loads) # remove extra spaces
      gsub("^(require|library)\\(['\"]?([^,'\"]+)['\"]?(,.+)?\\).*$", "\\2", loads)
    }))
  return(unique(pkgs))
}

#'
#' Retrieves all subsequent dependencies for packages described by version
#' object passed.
#'
#' @param vers version object describing packages to retrieve all subsequent
#'    dependencies for.
#' @param repo_info description of repository to search for
#'    dependencies in. Unused if avail_pkgs passed. (object of rsuite_repo_info)
#' @param type type of packages to search. . Unused if avail_pkgs passed.
#' @param all_pkgs matrix same as available.packages return.
#'
#' @return pkgSearchResult object containing packages from vers and all their
#'    subsequent dependencies.
#'
#' @keywords internal
#' @noRd
#'
collect_all_subseq_deps <- function(vers, repo_info, type, all_pkgs = NULL, extra_reqs = NULL) {
  stopifnot(is.versions(vers))
  stopifnot(is.null(extra_reqs) || is.versions(extra_reqs))

  if (is.null(all_pkgs)) {
    stopifnot(!missing(repo_info))
    stopifnot(!missing(type))

    contrib_url <- repo_info$get_contrib_url(type)    # from 53_repositories.R
    avail_vers <- vers.collect(contrib_url)
    all_pkgs <- avail_vers$get_avails()
  } else {
    avail_pkgs <- as.data.frame(all_pkgs, stringsAsFactors = FALSE)
    avail_vers <- vers.collect(pkgs = avail_pkgs)
  }

  vers <- vers.rm_base(vers)
  vers_cr <- vers.check_against(vers, avail_vers, extra_reqs)

  next_cr <- vers_cr
  while (check_res.has_found(next_cr)) {
    dep_avails <- vers.pick_available_pkgs(check_res.get_found(next_cr))

    dep_vers <- vers.from_deps_in_avails(dep_avails)
    dep_vers <- vers.rm_base(dep_vers)

    next_cr <- vers.check_against(dep_vers, avail_vers, extra_reqs)
    vers_cr <- check_res.union(vers_cr, next_cr)
  }

  return(vers_cr)
}


#' Gets project dependencies lock verse, vmin and vmax are equal, because
#' we want to keep the locked version of the packages
#'
#' @param params project parameters. (type: rsuite_project_params)
#'
#' @return vers with locked environment dependencies
#'
#' @keywords internal
#' @noRd
#'
get_lock_env_vers <- function(params) {
  env_lock <- read.dcf(params$lock_path)
  env_lock_vers <- do.call("vers.union",
                           apply(X = env_lock, 1,
                                 FUN = function(pkg){
                                   vers.build(pkg["Package"], pkg["Version"], pkg["Version"])
                                 })) # from 60_versions.R
  return(env_lock_vers)
}


#' Locks the project dependencies.
#'
#' The function is also responsible for updating the env.lock file in the following
#' cases:
#'
#' - A new dependency was added (in that case the new dependency is appended to the
#' env.lock file)
#'
#' - An existing dependency was deleted (depending on the relock flag the funcion will
#' either return an error or update the env.lock file)
#'
#' - An existing dependency was updated e.g. RSuite was updated from version 0.26 to 0.27
#' (depending on the relock flag the funcion will either return an error or update the
#'  env.lock file)
#'
#'
#' @param avail_vers version object describing available project dependencies.
#' @param params project parameters (type" rsuite_project_params)
#' @param relock if TRUE allows to update the env.lock file
#'   (type: logical, default: FALSE)
#'
#' @keywords internal
#' @noRd
#'
lock_prj_deps <- function(avail_vers, params, relock = FALSE) {
  if (!file.exists(params$lock_path)) {
    return(avail_vers)
  }

  # remove project packages
  project_packages <- build_project_pkgslist(params$pkgs_path) # from 51_pkg_info.R
  env_lock_vers <- get_lock_env_vers(params) #from 52_dependencies.R
  env_lock_vers <- vers.rm(env_lock_vers, project_packages)

  avail_vers_locked <- vers.drop_avails(avail_vers)
  avail_vers_locked <- vers.union(avail_vers_locked, env_lock_vers)

  avail_pkgs <- vers.get_names(avail_vers)
  env_lock_pkgs <- vers.get_names(env_lock_vers)
  is_relocking_needed <- FALSE

  # look for new dependencies
  new_deps <- setdiff(avail_pkgs, env_lock_pkgs)
  if (length(new_deps) != 0) {
    pkg_loginfo("Following packages will be added to lock requirements: %s",
                paste(new_deps, collapse = ","))
    is_relocking_needed <- TRUE
  }

  # look for deleted dependencies
  deleted_deps <- setdiff(env_lock_pkgs, avail_pkgs)
  if (length(deleted_deps) != 0) {
    assert(any(relock),
           paste0("Following packages to be removed from lock requirements: %s.",
                  " Please, relock the project environment"),
           paste(deleted_deps, collapse = ","))

    pkg_loginfo("Following packages will be removed from lock requirements: %s",
                paste(deleted_deps, collapse = ","))
    avail_vers_locked <- vers.rm(avail_vers_locked, deleted_deps)
    is_relocking_needed <- TRUE
  }

  avail_vers_candidate <- vers.add_avails(avail_vers_locked, avail_vers$get_avails())

  # look for updated dependencies
  unfeasibles <- vers.get_unfeasibles(avail_vers_candidate)
  if (length(unfeasibles) != 0) {
    assert(any(relock),
           paste0("Locked environment requirements cannot be satisfied for: %s.",
                  " Please, relock the project environment"),
           paste(unfeasibles, collapse = ","))

    pkg_loginfo("Following packages will be updated in lock requirements: %s",
                paste(unfeasibles, collapse = ","))
    is_relocking_needed <- TRUE
  } else {
    # candidate is good
    avail_vers <- avail_vers_candidate
  }

  if (is_relocking_needed) {
    write.dcf(avail_vers$get_avails()[, c("Package", "Version")], file = params$lock_path)
  }

  return(avail_vers)
}
