#' Download Sentinel images from a search list
#'
#' \code{senDownSearch} downloads the images from a list of uniform resource
#' locators (URLs) generated by the \code{\link{senSearch}} function from ESA’s
#' `SciHub' web service. The images are saved as GTiff files in the \code{AppRoot}
#' directory.
#' 
#' \code{senDownSearch} downloads the list of URLs provided by 
#' \code{\link{senSearch}}. In case the process is interrupted, the image file
#' could be corrupted. The function detects the corrupted files and restarts the
#' process. To prevent the computer from crashing, there is a maximum number of
#' attempts to try to donwnload the image (\code{nattempts}). The default number
#' of attempts is set to 3. The function requires an ESA’s `SciHub' account, which
#' can be obtained 
#' \href{https://scihub.copernicus.eu/dhus/#/self-registration}{here}.
#'
#' @param searchres the output from the \code{\link{senSearch}} function.
#' @param username ESA’s `SciHub' username.
#' @param password ESA’s `SciHub' password.
#' @param AppRoot the directory where the outcoming time series are saved.
#' @param unzip logical argument. If \code{TRUE}, unzips the images.
#' @param overwrite logical argument. If \code{TRUE}, overwrites the existing
#' images with the same name.
#' @param nattempts the number of attempts to download an image in case it
#' becomes corrupted.
#' @param ... arguments for nested functions.
#'
#' @examples
#' \dontrun{
#' # load a spatial polygon object of Navarre
#' data(ex.navarre)
#' # download S2MSI1C products sensed by Sentinel-2
#' # in July-August 2018
#' searchres <- senSearch(startDate = as.Date("2018-07-29","%Y-%m-%d"),
#'                        endDate = as.Date("2018-08-06","%Y-%m-%d"),
#'                        platform = "Sentinel-2",
#'                        extent = ex.navarre,
#'                        product = "S2MSI1C",
#'                        username = "username",
#'                        password = "password")
#'
#' # filtering the path R094 where Navarre is located
#' names(searchres)
#' searchres.R094 <- searchres[grepl("R094", names(searchres))]
#' names(searchres.R094)
#' # list the dates in searchres
#' senGetDates(names(searchres.R094),format="%Y%j")
#' src <- paste0(tempdir(),"/Path_for_downloading_folder")
#' # donwload the imagery
#' senDownSearch(searchres = searchres.R094,
#'               username = "username",
#'               password = "password",
#'               AppRoot = src,
#'               unzip = TRUE)
#' src.unzip <- file.path(src,"unzip")
#' files<-list.files(src.unzip,
#'                   pattern = "\\TCI.jp2$",
#'                   full.names = TRUE,
#'                   recursive = TRUE)
#' rgb<-stack(files[1])
#' plotRGB(rgb)
#' }
senDownSearch<-function(searchres,
                        AppRoot,
                        username=NULL,
                        password=NULL,
                        nattempts = 5,
                        unzip=FALSE,
                        overwrite=FALSE,
                        ...){
  arg<-list(...)
  AppRoot<-pathWinLx(AppRoot)
  if(nattempts==0){
    message(paste0("Error downloading ",names(searchres)))
    return(NULL)
  }
  if(is.null(username)|is.null(password)){
    stop("Username and/or password not defined!")
  }
  downFolder<-file.path(AppRoot,"/raw")
  dir.create(downFolder,recursive=TRUE,showWarnings = FALSE)
  message(paste0("Downloading the images in: ",downFolder))
  if(unzip){
    unzipFolder<-file.path(AppRoot,"/unzip")
    dir.create(unzipFolder,recursive=TRUE,showWarnings = FALSE)
  }
  n.imgs<-length(searchres)
  c.handle = new_handle()
  handle_setopt(c.handle,
                referer=getRGISToolsOpt("SCIHUBHUSURL"),
                useragent = getRGISToolsOpt("USERAGENT"),
                followlocation = TRUE ,
                autoreferer = TRUE ,
                username=username,
                password=password)
  for(i in 1:n.imgs){
    url<-searchres[i]
    file.name<-names(url)
    tryCatch({
      downPath<-file.path(downFolder,paste0(file.name,".zip"))
      if((!file.exists(downPath))|overwrite){
        message(paste0("Downloading image ",file.name," (",i,"/",n.imgs,")"))
        image.url<-URLencode(url)
        
        curl_download(image.url, destfile=downPath,handle = c.handle)
      }

      #md5 check
      md5.url<-paste0(gsub("$value","",url,fixed = TRUE),"Checksum/Value/$value")
      message(md5.url)
      repeat{
        response<-curl(md5.url,handle =c.handle)
        md5.text<-readLines(response)
        if(!grepl("Error",md5.text)){
          message(paste0("Get md5: ",md5.text))
          break
        }else{
          message("md5 not found! trying again.")
          Sys.sleep(10)
        }
      }
      if(!genCheckMD5(downPath,oficial.md5=md5.text,...)){
        message(paste0("Error cheking ",file.name," file md5: ",md5.text))
        file.remove(downPath)
        senDownSearch(username=username,
                      password=password,
                      searchres=searchres[i],
                      unzip=unzip,
                      overwrite = overwrite,
                      nattempts=nattempts -1,
                      AppRoot = AppRoot,
                      ...)
      }else{
        message(paste0("OK: cheking ",file.name," file md5."))
        if(unzip){
          message("Unzipping ", basename(downPath)," file.")
          unzip(zipfile=downPath,
                exdir = unzipFolder,
                overwrite=overwrite)
        }
      }
    }, error = function(e) {
      message(paste0("ERROR:",e))
      file.remove(downPath)
      senDownSearch(username=username,
                    password=password,
                    searchres=searchres[i],
                    unzip=unzip,
                    overwrite = overwrite,
                    nattempts=nattempts -1,
                    AppRoot = AppRoot,
                    ...)
    }, finally = {
    })
  }

  if(unzip){
    message(paste0("The images have been unzipped in: ",unzipFolder))
  }else{
    message(paste0("The images have been downloaded and saved on HDD. \nFile path: ",downFolder))
  }
  rm(c.handle)
}
