#' @title
#' Extract a single data tibble from a REDCapTidieR supertibble
#'
#' @description
#' Take a supertibble generated with `read_redcap()`
#' and return one of its data tibbles.
#'
#' @details
#' This function makes it easy to extract a single instrument's data from a
#' REDCapTidieR supertibble.
#'
#' @returns A `tibble`.
#'
#' @param supertbl A supertibble generated by `read_redcap()`. Required.
#' @param tbl The `redcap_form_name` of the data tibble to extract. Required.
#'
#' @importFrom checkmate assert_character
#' @importFrom cli cli_abort
#' @importFrom tidyselect all_of
#'
#' @examples
#' # Mock up a supertibble
#' sample_data <- tibble::tribble(
#'   ~redcap_form_name,    ~redcap_data,   ~structure,
#'   "super_hero_powers",  list(),         "repeating",
#'   "heroes_information", list(),         "nonrepeating"
#' )
#'
#' extract_tibble(sample_data, "heroes_information")
#'
#' @export

extract_tibble <- function(supertbl,
                           tbl) {
  # Check tbl is valid ----
  assert_character(tbl)

  if (length(tbl) > 1) {
    cli_abort("Only one table may be supplied.")
  }

  # Extract specified table ----
  out <- extract_tibbles(supertbl, tbls = all_of(tbl))[[1]]

  out
}


#' Extract data tibbles from a REDCapTidieR supertibble into a list
#'
#' @description
#' Take a supertibble generated with `read_redcap()`
#' and return a named list of data tibbles.
#'
#' @details
#' This function makes it easy to extract a multiple instrument's data from a
#' REDCapTidieR supertibble into a named list. Specifying instruments using
#' tidyselect helper functions such as `dplyr::starts_with()`
#' or `dplyr::ends_with()` is supported.
#'
#' @returns A named list of `tibble`s
#'
#' @param supertbl A supertibble generated by `read_redcap()`. Required.
#' @param tbls A vector of `form_name`s or a tidyselect helper. Default is
#' `dplyr::everything()`.
#'
#' @importFrom rlang enquo
#' @importFrom dplyr select %>%
#' @importFrom tidyselect eval_select everything
#' @importFrom tidyr pivot_wider
#' @importFrom purrr map pluck
#'
#' @examples
#' # Mock up a supertibble
#' sample_data <- tibble::tribble(
#'   ~redcap_form_name,    ~redcap_data, ~structure,
#'   "super_hero_powers",  list(),       "repeating",
#'   "heroes_information", list(),       "nonrepeating"
#' )
#'
#' # Extract all data tibbles
#' extract_tibbles(sample_data)
#'
#' # Only extract data tibbles starting with "heroes"
#' extract_tibbles(sample_data, starts_with("heroes"))
#'
#' @export

extract_tibbles <- function(supertbl,
                            tbls = everything()) {
  # Extract specified table ----
  # Pass tbls as an expression for enquosure
  tbls <- enquo(tbls)

  out <- supertbl %>%
    select("redcap_form_name", "redcap_data") %>%
    pivot_wider(
      names_from = "redcap_form_name",
      values_from = "redcap_data"
    )

  out <- out[eval_select(tbls, data = out)]

  out %>%
    map(.f = ~ pluck(.)[[1]])
}
