# Currently this function could only parse svg file created by
# CairoSVG() in Cairo package
parseSVG = function(file.name)
{
	# Use XML package
    svgFile = xmlParse(file.name);
	# Don't forget the name space!
    newXMLNamespace(xmlRoot(svgFile), "http://www.w3.org/2000/svg", "svg");
	# Find the first <g> child of <svg>
    pathRoot = getNodeSet(svgFile, "/svg:svg/svg:g");
	if(!length(pathRoot)) stop(sprintf("Failed in parsing file '%s'",
									   file.name));
	pathRoot = pathRoot[[1]];
    
	# Default style for a <path> node
    defaultStyle = c("stroke" = "none",
                    "stroke-width" = "1",
                    "stroke-linecap" = "butt",
                    "stroke-linejoin" = "miter",
                    "stroke-miterlimit" = "4",
                    "stroke-opacity" = "1",
                    "fill" = "rgb(0%,0%,0%)",
                    "fill-rule" = "nonzero",
                    "fill-opacity" = "1");
    
    # Handle <path> style in named vector
    parseStyle = function(style)
    {
        if(is.null(style)) return(NULL);
        s = unlist(strsplit(style, ";"));
        val = strsplit(s, ":");
        result = sapply(val, function(x) x[2]);
        names(result) = sapply(val, function(x) x[1]);
        return(result);
    }
    # Update the attributes in "old" style with the values in "new"
    # "old" must contain "new"
    updateStyle = function(old, new)
    {
        if(is.null(new)) return(old);
        result = old;
        result[names(new)] = new;
        return(result);
    }
    # Iteratively update the style from parent nodes
    updateStyleUpward = function(node)
    {
        style = xmlAttrs(node)["style"];
        if(is.na(style)) style = NULL;
        style = parseStyle(style);
        style = updateStyle(defaultStyle, style);
        parentNode = xmlParent(node);
        # Recursively search the parent
        while(!is.null(parentNode))
        {
            parentStyle = xmlAttrs(parentNode)["style"];
            if(is.null(parentStyle) || is.na(parentStyle)) parentStyle = NULL;
            parentStyle = parseStyle(parentStyle);
            style = updateStyle(style, parentStyle);
            parentNode = xmlParent(parentNode);
        }
        return(style);
    }
    # Parse <path> and <use> nodes into structured lists
	#
	# <path style="" d="">   =====>   style=..., d=..., x=0, y=0
	#
	# <use xlink:href="#glyph0-0" x="63.046875" y="385.921875"/>
	# =====>
	# style=..., d=..., x=63.046875, y=385.921875
	#
    parseNode = function(node)
    {
        if(xmlName(node) == "use")
        {
            attrs = xmlAttrs(node);
            refID = sub("#", "", attrs["href"]);
            refPathNode = getNodeSet(svgFile, sprintf("//*[@id='%s']/svg:path", refID))[[1]];
            style = updateStyleUpward(refPathNode);
            style = updateStyle(style, updateStyleUpward(node));
            d = xmlAttrs(refPathNode)["d"];
            x = xmlAttrs(node)["x"];
            y = xmlAttrs(node)["y"];
        } else if(xmlName(node) == "path") {
            style = updateStyleUpward(node);
            d = xmlAttrs(node)["d"];
            x = y = 0;
        } else stop("Unknown child node of '/svg/g'");
        xy = as.numeric(c(x, y));
        names(d) = NULL;
        names(xy) = NULL;
        return(list(style = style, d = d, xy = xy));
    }
    # Flatten nodes
	# <g>
	#   <use />
	#   <use />
	#   <use />
	# </g>
	#
	# =====>
	#
	# <use />
	# <use />
	# <use />
    expandNode = function(node)
    {
        children = xmlChildren(node);
        res = if(!length(children)) node else children;
        return(res);
    }
    nodes = unlist(xmlSApply(pathRoot, expandNode));
    names(nodes) = NULL;
    paths = lapply(nodes, parseNode);
    return(paths);
}


##' Convert a sequence of SVG files to SWF file
##'
##' Given the file names of a sequence of SVG files, this function could
##' convert them into a Flash file (.swf).
##'
##' This function uses the XML package in R and a subset of librsvg
##' (\url{http://librsvg.sourceforge.net/}) to parse the SVG file, and
##' utilize the Ming library (\url{http://www.libming.org/}) to
##' implement the conversion. Currently the support for SVG file is
##' \strong{NOT} perfect and \strong{ONLY} support files generated by
##' \code{\link[Cairo]{CairoSVG}()} in the \pkg{Cairo} package.
##'
##' @param input the file names of the SVG files to be converted
##' @param output the name of the output SWF file
##' @param bgColor background color of the output SWF file
##' @param interval the time interval (in seconds) between animation frames
##' @return The path of the generated SWF file if successful.
##' @note Currently there is no function in R that could auto-generate
##' a sequence of SVG files, so you may create the them once at a time.
##' See the example.
##' @export
##' @author Yixuan Qiu <\email{yixuan.qiu@@cos.name}>
##' @examples \dontrun{if(require("Cairo")) {
##'   olddir = setwd(tempdir())
##'   filenames = sprintf("Rplot%03d.svg", 1:9)
##'   for(fn in filenames) {
##'     CairoSVG(fn)
##'     plot(runif(20), ylim = c(0, 1))
##'     dev.off()
##'   }
##'   output = svg2swf(filenames, bgColor = "steelblue")
##'   swf2html(output)
##'   setwd(olddir)
##' }
##' }
##'
svg2swf = function(input, output = "./movie.swf", bgColor = "white",
				   interval = 1)
{
    if(!inherits(input, "character")) 
        stop("'input' must be a character vector naming the input SVG files");
	
	outDir = dirname(output);
	outFile = basename(output);
	bg = col2rgb(bgColor, alpha = FALSE);
	bg = as.integer(bg);

	filesData = lapply(input, parseSVG);
    firstFile = xmlParse(input[1]);
    size = xmlAttrs(xmlRoot(firstFile))["viewBox"];
    size = as.numeric(unlist(strsplit(size, " ")));

	oldwd = setwd(outDir);
    .Call("svg2swf", filesData, as.character(outFile), size,
		  bg, as.numeric(interval), PACKAGE = "R2SWF");
    setwd(oldwd);

	output = normalizePath(output);
	cat("SWF file created at ", output, ".\n", sep = "");
	invisible(output);
}

