% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MIM.points2.R
\name{MIM.points2}
\alias{MIM.points2}
\title{QTL Short Distance Correction by MIM with Selective Genotyping}
\usage{
MIM.points2(
  QTL,
  marker,
  geno,
  y,
  yu = NULL,
  sele.g = "n",
  tL = NULL,
  tR = NULL,
  method = "EM",
  type = "RI",
  D.matrix = NULL,
  ng = 2,
  cM = TRUE,
  scope = 5,
  speed = 1,
  crit = 10^-3,
  console = TRUE
)
}
\arguments{
\item{QTL}{matrix. A q*2 matrix contains the QTL information, where
the row dimension q is the number of QTLs in the chromosomes. The
first column labels the chromosomes where the QTLs are located, and
the second column labels the positions of QTLs (in morgan (M) or
centimorgan (cM)). Note that chromosomes and positions must be divided
in order.}

\item{marker}{matrix. A k*2 matrix contains the marker information,
where the row dimension k is the number of markers in the chromosomes.
The first column labels the chromosomes where the markers are located,
and the second column labels the positions of QTLs (in morgan (M) or
centimorgan (cM)). Note that chromosomes and positions must be divided
in order.}

\item{geno}{matrix. A n*k matrix contains the k markers of the n
individuals. The marker genotypes of P1 homozygote (MM),
heterozygote (Mm), and P2 homozygote (mm) are coded as 2, 1, and 0,
respectively, and NA for missing value.}

\item{y}{vector. A vector that contains the phenotype values of
individuals with genotypes.}

\item{yu}{vector. A vector that contains the phenotype value
of the individuals without genotypes.}

\item{sele.g}{character. If sele.g="n", it will consider that the
data is not a selective genotyping data but the complete genotyping
data. If sele.g="p", it will consider that the data is a selective
genotyping data, and use the proposed model (Lee 2014) to analyze.
If sele.g="t", it will consider that the data is a selective
genotyping data, and use the truncated model (Lee 2014) to analyze.
If sele.g="f, it will consider that the data is a selective
genotyping data, and use the frequency-based model (Lee 2014) to
analyze. Note that the yu must be input when sele.g="p" of "f".}

\item{tL}{numeric. The lower truncation point of phenotype value
when sele.g="t". Note that when sele.g="t" and tL=NULL, the yu
must be input and the function will consider the minimum of yu
as the lower truncation point.}

\item{tR}{numeric. The upper truncation point of phenotype value
when sele.g="t". Note that when sele.g="t" and tR=NULL, the yu
must be input and the function will consider the maximum of yu
as the upper truncation point.}

\item{method}{character. method="EM" means the interval mapping method
by Lander and Botstein (1989) is used in the analysis, while
method="REG" means  the approximate regression interval mapping method
by Haley and Knott (1992) is considered in the analysis.}

\item{type}{character. The population type of the dataset. Include
backcross (type="BC"), advanced intercross population (type="AI"), and
recombinant inbred population (type="RI").}

\item{D.matrix}{matrix. The design matrix of QTL effects is a g*p
matrix, where g is the number of possible QTL genotypes, and p is the
number of effects considered in the MIM model. The design matrix can
be easily generated by the function D.make(). If being NULL, it Will
automatically generate a design matrix with all additive and
dominant effects and without any epistasis effect.}

\item{ng}{integer. The generation number of the population type. For
example, the BC1 population is type="BC" with ng=1; the AI F3
population is type="AI" with ng=3.}

\item{cM}{logical. Specify the unit of marker position. cM=TRUE for
centimorgan. Or cM=FALSE for morgan.}

\item{scope}{numeric vector. The search scope of each QTL. In the
MIM process, it will search forward and backward for the corresponding
cM. User can assign a numeric number for every QTL or a numeric vector
for each QTL. Note that 0 denote that the corresponding QTL position
is fixed, and the position of its surrounding positions will not be
searched.}

\item{speed}{numeric. The walking speed of the QTL search (in cM).}

\item{crit}{numeric. The convergence criterion of the EM algorithm.
The E and M steps will be iterated until a convergence criterion
is satisfied. It must be between 0 and 1.}

\item{console}{logical. To decide whether the process of the algorithm
will be shown in the R console or not.}
}
\value{
\item{effect}{The estimated effects, log likelihood value, and LRT
statistics of all searched positions.}
\item{QTL.best}{The positions of the best QTL combination.}
\item{effect.best}{The estimated effects and LRT statistics of the best
QTL combination.}
\item{model}{The model of selective genotyping data in this analysis.}
}
\description{
Expectation-maximization algorithm for QTL multiple interval mapping.
Find the best QTL position near the designated QTL position. This
function can handle the genotype data witch is selective genotyping.
}
\examples{
# load the example data
load(system.file("extdata", "exampledata.RDATA", package = "QTLEMM"))

# make the seletive genotyping data
ys <- y[y > quantile(y)[4] | y < quantile(y)[2]]
yu <- y[y >= quantile(y)[2] & y <= quantile(y)[4]]
geno.s <- geno[y > quantile(y)[4] | y < quantile(y)[2],]

# run and result
result <- MIM.points2(QTL, marker, geno.s, ys, yu, sele.g = "p",
 type = "RI", ng = 2, scope = c(0,3,0), speed = 2)
result$QTL.best
result$effect.best
}
\references{
KAO, C.-H. and Z.-B. ZENG 1997 General formulas for obtaining the maximum
likelihood estimates and the asymptotic variance-covariance matrix in QTL
mapping when using the EM algorithm. Biometrics 53, 653-665.

KAO, C.-H., Z.-B. ZENG and R. D. TEASDALE 1999 Multiple interval mapping
for Quantitative Trait Loci. Genetics 152: 1203-1216.

H.-I LEE, H.-A. HO and C.-H. KAO 2014 A new simple method for improving
QTL mapping under selective genotyping. Genetics 198: 1685-1698.
}
\seealso{
\code{\link[QTLEMM]{EM.MIM2}}
\code{\link[QTLEMM]{MIM.points}}
}
