% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/svds.R
\name{svds}
\alias{svds}
\title{Find a few singular values and vectors on large, sparse matrix}
\usage{
svds(
  A,
  NSvals,
  which = "L",
  tol = 1e-06,
  u0 = NULL,
  v0 = NULL,
  orthou = NULL,
  orthov = NULL,
  prec = NULL,
  isreal = NULL,
  ...
)
}
\arguments{
\item{A}{matrix or a function with signature f(x, trans) that returns
\code{A \%*\% x} when \code{trans == "n"} and
\code{t(Conj(A)) \%*\% x} when \code{trans == "c"}.}

\item{NSvals}{number of singular triplets to seek.}

\item{which}{which singular values to find:
\describe{
   \item{\code{"L"}}{the largest values;}
   \item{\code{"S"}}{the smallest values;}
   \item{vector of numbers}{the closest values to these points.}
}}

\item{tol}{a triplet \eqn{(\sigma,u,v)} is marked as converged when
\eqn{\sqrt{\|Av - \sigma u\|^2+\|A^*u - \sigma v\|^2} \le tol\|A\|}{sqrt(||A*v - sigma*u||^2 + ||A'*u - \sigma*v||^2}
is smaller than \eqn{tol*||A||}, or close to the minimum tolerance that
the selected method can achieve.}

\item{u0}{matrix whose columns are educated guesses of the left singular
vectors to find.}

\item{v0}{matrix whose columns are educated guesses of the right singular
vectors to find.}

\item{orthou}{find left singular vectors orthogonal to the space spanned by
the columns of this matrix; useful to avoid finding some triplets or
to find new solutions.}

\item{orthov}{find right singular vectors orthogonal to the space spanned by
the columns of this matrix.}

\item{prec}{preconditioner used to accelerated the convergence; it is a named
list of matrices or functions such as \code{solve(prec[[mode]],x)} or
\code{prec[[mode]](x)} return an approximation of \eqn{OP^{-1} x},
where
\tabular{cc}{
  \code{mode}  \tab \eqn{OP} \cr
  \code{"AHA"} \tab \eqn{A^*A} \cr
  \code{"AAH"} \tab \eqn{A A^*} \cr
  \code{"aug"} \tab \eqn{[0 A; A^* 0]}
}
The three values haven't to be set. It is recommended to set
\code{"AHA"} for matrices with nrow > ncol; \code{"AAH"} for
matrices with nrow < ncol; and additionally \code{"aug"} for
\code{tol} < 1e-8.}

\item{isreal}{whether A \%*\% x always returns real number and not complex.}

\item{...}{other PRIMME options (see details).}
}
\value{
list with the next elements
   \describe{
      \item{\code{d}}{the singular values \eqn{\sigma_i}}
      \item{\code{u}}{the left singular vectors \eqn{u_i}}
      \item{\code{v}}{the right singular vectors \eqn{v_i}}
      \item{\code{rnorms}}{the residual vector norms
         \eqn{\sqrt{\|Av - \sigma u\|^2+\|A^*u - \sigma v\|^2}}{sqrt(||A*v - sigma*u||^2 + ||A'*u - \sigma*v||^2}}
      \item{\code{stats$numMatvecs}}{matrix-vector products performed}
      \item{\code{stats$numPreconds}}{number of preconditioner applications performed}
      \item{\code{stats$elapsedTime}}{time expended by the eigensolver}
      \item{\code{stats$timeMatvec}}{time expended in the matrix-vector products}
      \item{\code{stats$timePrecond}}{time expended in applying the preconditioner}
      \item{\code{stats$timeOrtho}}{time expended in orthogonalizing}
      \item{\code{stats$estimateANorm}}{estimation of the norm of A}
   }
}
\description{
Compute a few singular triplets from a specified region (the largest, the
smallest, the closest to a point) on a matrix using PRIMME [1].
Only the matrix-vector product of the matrix is required. The used method is
usually faster than a direct method (such as \code{\link{svd}}) if
seeking few singular values and the matrix-vector product is cheap. For
accelerating the convergence consider to use preconditioning  and/or
educated initial guesses.
}
\details{
Optional arguments to pass to PRIMME eigensolver (see further details at
[2]):

\describe{
   \item{\code{aNorm}}{estimation of norm-2 of A, used in convergence test
      (if not provided, it is estimated as the largest eigenvalue in 
      magnitude seen)}
   \item{\code{maxBlockSize}}{maximum block size (like in subspace iteration
      or LOBPCG)}
   \item{\code{printLevel}}{message level reporting, from 0 (no output) to 5
      (show all)} 
   \item{\code{locking}}{1, hard locking; 0, soft locking}
   \item{\code{maxBasisSize}}{maximum size of the search subspace}
   \item{\code{minRestartSize}}{ minimum Ritz vectors to keep in restarting}
   \item{\code{maxMatvecs}}{ maximum number of matrix vector multiplications}
   \item{\code{iseed}}{ an array of four numbers used as a random seed}
   \item{\code{method}}{which equivalent eigenproblem to solve
      \describe{
         \item{\code{"primme_svds_normalequation"}}{\eqn{A^*A} or \eqn{AA^*}}
         \item{\code{"primme_svds_augmented"}}{ \eqn{[0 A^*;A 0]}}
         \item{\code{"primme_svds_hybrid"}}{ first normal equations and
                     then augmented (default)}
      }                   
   }
   \item{\code{locking}}{1, hard locking; 0, soft locking}
   \item{\code{primmeStage1, primmeStage2}}{list with options for the first
      and the second stage solver; see \code{\link{eigs_sym}}}
}

If \code{method} is \code{"primme_svds_normalequation"}, the minimum
tolerance that can be achieved is \eqn{\|A\|\epsilon/\sigma}, where \eqn{\epsilon}
is the machine precision. If \code{method} is \code{"primme_svds_augmented"}
or \code{"primme_svds_hybrid"}, the minimum tolerance is \eqn{\|A\|\epsilon}.
However it may not return triplets with singular values smaller than
\eqn{\|A\|\epsilon}.
}
\examples{
A <- diag(1:5,10,5)  # the singular values of this matrix are 1:10 and the
                        # left and right singular vectors are the columns of
                        # diag(1,100,10) and diag(10), respectively
r <- svds(A, 3);
r$d # the three largest singular values on A
r$u # the corresponding approximate left singular vectors
r$v # the corresponding approximate right singular vectors
r$rnorms # the corresponding residual norms
r$stats$numMatvecs # total matrix-vector products spend

r <- svds(A, 3, "S") # compute the three smallest values

r <- svds(A, 3, 2.5) # compute the three closest values to 2.5

A <- diag(1:500,500,100)   # we use a larger matrix to amplify the difference
r <- svds(A, 3, 2.5, tol=1e-3); # compute the values with 
r$rnorms                               # residual norm <= 1e-3*||A||

# Build the diagonal squared preconditioner
# and see how reduce the number matrix-vector products
P <- diag(colSums(A^2))
svds(A, 3, "S", tol=1e-3)$stats$numMatvecs
svds(A, 3, "S", tol=1e-3, prec=list(AHA=P))$stats$numMatvecs

# Passing A and the preconditioner as functions
Af <- function(x,mode) if (mode == "n") A\%*\%x else crossprod(A,x);
P = colSums(A^2);
PAHAf <- function(x) x / P;
r <- svds(Af, 3, "S", tol=1e-3, prec=list(AHA=PAHAf), m=500, n=100)

# Passing initial guesses
v0 <- diag(1,100,4) + matrix(rnorm(400), 100, 4)/100;
svds(A, 4, "S", tol=1e-3)$stats$numMatvecs
svds(A, 4, "S", tol=1e-3, v0=v0)$stats$numMatvecs

# Passing orthogonal constrain, in this case, already compute singular vectors
r <- svds(A, 4, "S", tol=1e-3); r$d
svds(A, 4, "S", tol=1e-3, orthov=r$v)$d

}
\references{
[1]  L. Wu, E. Romero and A. Stathopoulos, \emph{PRIMME_SVDS: A High-Performance
     Preconditioned SVD Solver for Accurate Large-Scale Computations},
     J. Sci. Comput., Vol. 39, No. 5, (2017), S248--S271.

[2] \url{http://www.cs.wm.edu/~andreas/software/doc/svdsc.html#parameters-guide}
}
\seealso{
\code{\link{svd}} for computing all singular triplets;
\code{\link{eigs_sym}} for computing a few eigenvalues and vectors
   from a symmetric/Hermitian matrix.
}
