% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ODEmorris.default.R
\name{ODEmorris.default}
\alias{ODEmorris.default}
\title{Morris Screening for General ODE Models}
\usage{
\method{ODEmorris}{default}(mod, pars, state_init, times, binf = 0,
  bsup = 1, r = 500, design = list(type = "oat", levels = 10, grid.jump =
  1), scale = TRUE, ode_method = "lsoda", parallel_eval = FALSE,
  parallel_eval_ncores = NA, ...)
}
\arguments{
\item{mod}{[\code{function(Time, State, Pars)}]\cr
model to examine, supplied in the manner as needed for 
\code{\link[deSolve]{ode}} (see example below).}

\item{pars}{[\code{character(k)}]\cr
names of the parameters to be included as input variables in Morris 
screening.}

\item{state_init}{[\code{numeric(z)}]\cr
vector of \code{z} initial values. Must be named (with unique names).}

\item{times}{[\code{numeric}]\cr
points of time at which the sensitivity analysis should be executed (vector
of arbitrary length). The first point of time must be greater than zero.}

\item{binf}{[\code{character(1} or \code{k)}]\cr
vector of lower borders of possible input parameter values.
If they are all equal, a single value can be set.}

\item{bsup}{[\code{character(1} or \code{k)}]\cr
vector of upper borders of possible input parameter values.
If they are all equal, a single value can be set.}

\item{r}{[\code{integer(1 or 2)}]\cr
if of length 1, the number of repetitions of the \code{design}. If of 
length 2, a space-filling optimization of the sampling design is used, see 
\code{\link[sensitivity]{morris}}. However, this space-filling optimization
might lead to long runtimes, so length 1 is recommended for \code{r}. 
Defaults to 500.}

\item{design}{[\code{list}]\cr
a list specifying the design type and its parameters,
cf. \code{\link[sensitivity]{morris}}.}

\item{scale}{[\code{logical(1)}]\cr
if \code{TRUE}, scaling is done for the input design of experiments after 
building the design and before calculating the elementary effects,
cf. \code{\link[sensitivity]{morris}}. Defaults to \code{TRUE}, which is
highly recommended if the factors have different orders of magnitude, see
\code{\link[sensitivity]{morris}}.}

\item{ode_method}{[\code{character(1)}]\cr
method to be used for solving the differential equations, see 
\code{\link[deSolve]{ode}}. Defaults to \code{"lsoda"}.}

\item{parallel_eval}{[\code{logical(1)}]\cr
logical indicating if the evaluation of the ODE model shall be performed
parallelized.}

\item{parallel_eval_ncores}{[\code{integer(1)}]\cr
number of processor cores to be used for parallelization. Only applies if
\code{parallel_eval = TRUE}. If set to \code{NA} (as per default) and 
\code{parallel_eval = TRUE}, 1 processor core is used.}

\item{...}{further arguments passed to or from other methods.}
}
\value{
List of class \code{ODEmorris} of length \code{length(state_init)} 
  containing in each element a matrix for one state variable. The
  matrices themselves contain the Morris screening results for all timepoints 
  (rows: \code{mu, mu.star} and \code{sigma} for every parameter; columns: 
  timepoints).
}
\description{
\code{ODEmorris.default} is the default method of \code{\link{ODEmorris}}. It
performs a sensitivity analysis for general ODE models using the Morris 
screening method.
}
\details{
Function \code{\link[deSolve]{ode}} from \code{\link[deSolve]{deSolve}} is 
  used to solve the ODE system.
  
  The sensitivity analysis is done for all state variables and all
  timepoints simultaneously using \code{\link[sensitivity]{morris}} from the 
  package \code{\link[sensitivity]{sensitivity}}.
  
  For non-ODE models, values for \code{r} are typically between 10 and 50.
  However, much higher values are recommended for ODE models (the default is
  \code{r = 500}).
}
\note{
If the evaluation of the model function takes too long, it might be helpful 
  to try another ODE-solver (argument \code{ode_method}). The 
  \code{ode_method}s \code{"vode"}, \code{"bdf"}, \code{"bdf_d"}, 
  \code{"adams"}, \code{"impAdams"} and \code{"impAdams_d"} might be faster 
  than the default \code{"lsoda"}.
  
  If \code{\link[sensitivity]{morris}} throws a warning message stating
  "In ... keeping ... repetitions out of ...", try using a bigger number of 
  \code{levels} in the \code{design} argument (only possible for OAT design).
}
\examples{
##### Lotka-Volterra equations #####
# The model function:
LVmod <- function(Time, State, Pars) {
  with(as.list(c(State, Pars)), {
    Ingestion    <- rIng  * Prey * Predator
    GrowthPrey   <- rGrow * Prey * (1 - Prey/K)
    MortPredator <- rMort * Predator
    
    dPrey        <- GrowthPrey - Ingestion
    dPredator    <- Ingestion * assEff - MortPredator
    
    return(list(c(dPrey, dPredator)))
  })
}
# The parameters to be included in the sensitivity analysis and their lower 
# and upper boundaries:
LVpars  <- c("rIng", "rGrow", "rMort", "assEff", "K")
LVbinf <- c(0.05, 0.05, 0.05, 0.05, 1)
LVbsup <- c(1.00, 3.00, 0.95, 0.95, 20)
# The initial values of the state variables:
LVinit  <- c(Prey = 1, Predator = 2)
# The timepoints of interest:
LVtimes <- c(0.01, seq(1, 50, by = 1))
# Morris screening:
set.seed(7292)
# Warning: The following code might take very long!
\donttest{
LVres_morris <- ODEmorris(mod = LVmod,
                          pars = LVpars,
                          state_init = LVinit,
                          times = LVtimes,
                          binf = LVbinf,
                          bsup = LVbsup,
                          r = 500,
                          design = list(type = "oat", 
                                        levels = 10, grid.jump = 1),
                          scale = TRUE,
                          ode_method = "lsoda",
                          parallel_eval = TRUE,
                          parallel_eval_ncores = 2)
}

##### FitzHugh-Nagumo equations (Ramsay et al., 2007) #####
FHNmod <- function(Time, State, Pars) {
  with(as.list(c(State, Pars)), {
    
    dVoltage <- s * (Voltage - Voltage^3 / 3 + Current)
    dCurrent <- - 1 / s *(Voltage - a + b * Current)
    
    return(list(c(dVoltage, dCurrent)))
  })
}
# Warning: The following code might take very long!
\donttest{
FHNres_morris <- ODEmorris(mod = FHNmod,
                           pars = c("a", "b", "s"),
                           state_init = c(Voltage = -1, Current = 1),
                           times = seq(0.1, 50, by = 5),
                           binf = c(0.18, 0.18, 2.8),
                           bsup = c(0.22, 0.22, 3.2),
                           r = 500,
                           design = list(type = "oat", 
                                         levels = 50, grid.jump = 1),
                           scale = TRUE,
                           ode_method = "adams",
                           parallel_eval = TRUE,
                           parallel_eval_ncores = 2)
}

}
\references{
J. O. Ramsay, G. Hooker, D. Campbell and J. Cao, 2007,
  \emph{Parameter estimation for differential equations: a generalized 
  smoothing approach}, Journal of the Royal Statistical Society, Series B, 
  69, Part 5, 741--796.
}
\seealso{
\code{\link[sensitivity]{morris}, \link{plot.ODEmorris}}
}
\author{
Stefan Theers, Frank Weber
}
