#' Natural Language Processing for Meta Analysis
#'
#' The \pkg{MetaNLP} package provides methods to quickly transform a
#' CSV-file with titles and abstracts to an R data frame that can be
#' used for automatic title-abstract screening using machine learning.
#'
#'
#' @import methods
#' @name MetaNLP
"_PACKAGE"


#' Create a data frame with document-term matrix
#'
#' A \code{MetaNLP} object is the base class of the package \pkg{MetaNLP}.
#' It is initialized by passing the path to a CSV file and constructs
#' a data frame whose column names are the words that occur in the titles
#' and abstracts and whose cells contain the word frequencies for each
#' paper.
#'
#' @rdname MetaNLP
setClass("MetaNLP", representation(data_frame = "data.frame"))

#' @param file Either the path to the CSV file or a data frame containing the
#' abstracts
#' @param bounds An integer vector of length 2. The first value specifies
#' the minimum number of appearances of a word to become a column of the word
#' count matrix, the second value specifies the maximum number.
#' Defaults to \code{c(2, Inf)}.
#' @param word_length An integer vector of length 2. The first value specifies
#' the minimum number of characters of a word to become a column of the word
#' count matrix, the second value specifies the maximum number.
#' Defaults to \code{c(3, Inf)}.
#' @param language The language for lemmatization and stemming. Supported
#' languages are \code{english}, \code{french}, \code{german}, \code{russian} and
#' \code{spanish}. For non-english languages make sure that the csv
#' which is processed has the correct encoding.
#' @param ... Additional arguments passed on to \code{read.csv2}, e.g. when
#' "," should be used as a separator or when the encoding should be changed.
#' See \link[utils]{read.table}.
#' @return An object of class \code{MetaNLP}
#'
#' @details
#' An object of class \code{MetaNLP} contains a slot data_frame where
#' the document-term matrix is stored as a data frame.
#' The CSV file must have a column \code{ID} to identify each paper, a column
#' \code{title} with the belonging titles of the papers and a column
#' \code{abstract} which contains the abstracts. If the CSV stores training data,
#' a column \code{decision} should exist, indicating whether an abstract
#' is included in the meta analysis. This column does not need to exist, because
#' there is no decision for test data yet. Allowed values in this column are
#' either "yes" and "no" or "include" and "exclude" or "maybe". The value "maybe"
#' is handled as a "yes"/"include".
#'
#' @examples
#' path <- system.file("extdata", "test_data.csv", package = "MetaNLP", mustWork = TRUE)
#' obj <- MetaNLP(path)
#'
#' @note
#' To ensure correct processing of the data when there are special characters
#' (e.g. "é" or "ü"), make sure that the csv-file is correctly encoded
#' as \code{UTF-8}.
#' The stemming algorithm makes use of the C libstemmer library generated by
#' Snowball. When german texts are stemmed, umlauts are replaced by their
#' non-umlaut equivalent, so "ä" becomes "a" etc.
#'
#' @rdname MetaNLP
#' @export
MetaNLP <- function(file,
                    bounds      = c(2, Inf),
                    word_length = c(3, Inf),
                    language    = "english",
                    ...) {
  title <- NULL
  abstract <- NULL

  # match language
  language <- match.arg(language, c("english", "french", "german",
                                    "russian", "spanish"), several.ok = FALSE)
  # get lemmatization dictionaries in correct language
  if(language != "english"){
    lexicon <- get0(language, envir = asNamespace("MetaNLP"))
  } else {
    lexicon <- lexicon::hash_lemmas
  }

  # load file
  if(is.character(file)) data <- utils::read.csv2(file, header = TRUE, ...)
  else data <- as.data.frame(file)

  # make column names lower case
  names(data) <- tolower(names(data))

  # check that all the necessary columns exist
  if(any(c(is.null(data$id), is.null(data$title), is.null(data$abstract)))) {
    stop("The columns 'id', 'title' and 'abstract' must exist!")
  }

  # only select rows without na values or empty string
  n_exclude <- nrow(subset(data, ((is.na(data$abstract) | data$abstract == "") |
                                (is.na(data$title) | data$title == ""))))
  data <-  subset(data, !((is.na(data$abstract) | data$abstract == "") |
                            (is.na(data$title) | data$title == "")))
  if(n_exclude > 0) {
    warning(paste(n_exclude, "row(s) was/were removed due to missing values!"))
  }

  suppressWarnings({data |>
    # select the columns "abstract" and "title"
    (`[`)(c("title", "abstract")) |>
    # add new column x where Title and Abstract are pasted
    within(x <- paste(title, abstract)) |>
    (`[[`)(c("x")) |>
    # lower case
    tolower() |>
    tm::VectorSource() |>
    # create corpus object
    tm::Corpus() |>
    # remove special characters
    tm::tm_map(tm::content_transformer(replaceSpecialChars), language = language) |>
    # strip white space
    tm::tm_map(tm::stripWhitespace) |>
    # lemmatization of the words
    tm::tm_map(textstem::lemmatize_strings, dictionary = lexicon) |>
    # only use word stems
    tm::tm_map(tm::stemDocument, language = language) |>
    # create matrix
    tm::TermDocumentMatrix(control = list(wordLengths = word_length)) |>
    as.matrix() |>
    t() |>
    as.data.frame() -> temp
  })

  # only choose word stems that appear at least a pre-specified number of times
  temp <- temp[, colSums(temp) >= bounds[1] & colSums(temp) <= bounds[2]]
#
  # order by column name
  index_vec <- order(names(temp))
  temp |>
    subset(select = index_vec) -> temp

  if(!is.null(data$decision)) {
    # allow for "maybe" as decision
    decision <- ifelse(data$decision %in% c("include", "maybe", "yes"),
                       "include", "exclude")

    # add columns containing the ids of the papers and the belonging decisions
    res <- cbind("id_" = data$id, "decision_" = decision, temp)
  } else {
    res <- cbind("id_" = data$id, temp)
  }

  return(new("MetaNLP", data_frame = res))
}


setMethod("print", signature("MetaNLP"),
          function(x) {
            sprintf("MetaNLP<nrow=%i,ncol=%i>",
                    nrow(x@data_frame), ncol(x@data_frame))
          })

setMethod("show", signature("MetaNLP"),
          function(object) {
            cat(print(object))
          })


#' Create word cloud from MetaNLP-object
#'
#' This method creates a word cloud from a MetaNLP object. The word size
#' indicates the frequency of the words.
#'
#' @param x A MetaNLP object to plot
#' @param y not used
#' @param max.words Maximum number of words in the word cloud
#' @param colors Character vector with the colors in
#' @param decision Stratify word cloud by decision. Default is no stratification.
#' @param ... Additional parameters for \link[wordcloud]{wordcloud}
#'
#' @examples
#' path <- system.file("extdata", "test_data.csv", package = "MetaNLP", mustWork = TRUE)
#' obj <- MetaNLP(path)
#' plt <- plot(obj)
#'
#' @return nothing
#' @export
setMethod("plot", signature("MetaNLP", y = "missing"),
          function(x,  y = NULL, max.words = 70,
                   colors = c("snow4", "darkgoldenrod1", "turquoise4", "tomato"),
                   decision = c("total", "include", "exclude"),
                   ...) {

            decision_ <- NULL
            dec <- match.arg(decision)
            # check whether decision column exists and filter data
            if(dec != "total") {
              if(is.null(x@data_frame$decision_)) {
                warning("Column decision_ does not exist. Word cloud is created by using the whole document-term matrix.")
                data <- x@data_frame
              }
              else {
                x@data_frame |>
                subset(decision_ == dec) -> data
              }
            }
            else data <- x@data_frame
            data$id_ <- NULL
            data$decision_ <- NULL

            # create word cloud
            words <- names(data)
            freqs <- colSums(data)

            wordcloud::wordcloud(words, freqs, max.words = max.words,
                                random.order = FALSE,
                                color = colors, ...)

          })
