% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BiCopEstMMD.R
\name{BiCopEstMMD}
\alias{BiCopEstMMD}
\title{Estimation of parametric bivariate copulas using
stochastic gradient descent on the MMD criteria}
\usage{
BiCopEstMMD(
  u1,
  u2,
  family,
  tau = 0.1,
  par = NULL,
  par2 = NULL,
  kernel = "gaussian",
  gamma = 0.23,
  alpha = 1,
  niter = 100,
  epsilon = 1e-04,
  method = "QMCV",
  quasiRNG = "sobol",
  ndrawings = 10
)
}
\arguments{
\item{u1}{vector of observations of the first coordinate, in \eqn{[0,1]}.}

\item{u2}{vector of observations of the second coordinate, in \eqn{[0,1]}.}

\item{family}{the chosen family of copulas
(see the documentation of the class \code{VineCopula::\link[VineCopula]{BiCop}()}
for the available families).}

\item{tau}{the copula family can be parametrized by the parameter \code{par}
or by Kendall's tau.
Here, the user can choose the initial value of tau for the stochastic gradient algorithm.}

\item{par}{if different from \code{NULL}, the parameter \code{tau} is ignored,
and the initial parameter must be given here.
The initial Kendall's tau is then computed thanks to
\code{VineCopula::\link[VineCopula]{BiCopPar2Tau}()}.}

\item{par2}{initial value for the second parameter, if any. (Works only for Student copula).}

\item{kernel}{the kernel used in the MMD distance:
 it can be a function taking in parameter \code{(u1, u2, v1, v2, gamma, alpha)}
 or a name giving the kernel to use in the list:
 \itemize{
   \item \code{gaussian}: Gaussian kernel \eqn{k(x,y) = \exp(-\|\frac{x-y}{\gamma}\|_2^2)
   }{k(x,y) = exp( - || (x-y) / gamma ||_2^2)}
   \item \code{exp.l2}: \eqn{k(x,y) = \exp(-\|\frac{x-y}{\gamma}\|_2)
   }{k(x,y) = exp( - || (x-y) / gamma ||_2)}
   \item \code{exp.l1}: \eqn{k(x,y) = \exp(-\|\frac{x-y}{\gamma}\|_1)
   }{k(x,y) = exp( - || (x-y) / gamma ||_1)}
   \item \code{inv.l2}: \eqn{k(x,y) = 1/(1+\|\frac{x-y}{\gamma}\|_2)^\alpha
   }{k(x,y) = 1 / ( 1 + || (x-y) / gamma ||_2 )^\alpha}
   \item \code{inv.l1}: \eqn{k(x,y) = 1/(1+\|\frac{x-y}{\gamma}\|_1)^\alpha
   }{k(x,y) = 1 / ( 1 + || (x-y) / gamma ||_1 )^\alpha}
 }
Each of these names can receive the suffix ".KG", such as "gaussian.KG"
to indicates that the kernel \eqn{k(x,y)} is replaced by
\eqn{k(\Phi^{-1}(x) , \Phi^{-1}(y))} where \eqn{\Phi^{-1}} denotes the quantile
function of the standard Normal distribution.}

\item{gamma}{parameter \eqn{\gamma} to be used in the kernel.}

\item{alpha}{parameter \eqn{\alpha} to be used in the kernel, if any.}

\item{niter}{number of iterations of the stochastic gradient algorithm.}

\item{epsilon}{the differential of \code{VineCopula::\link[VineCopula]{BiCopTau2Par}()}
is computed thanks to a finite difference with increment \code{epsilon}.}

\item{method}{the method of computing the stochastic gradient:
\itemize{
  \item \code{MC}: classical Monte-Carlo with \code{ndrawings} replications.
  \item \code{QMCV}: usual Monte-Carlo on U with \code{ndrawings} replications,
    quasi Monte-Carlo on V.
}}

\item{quasiRNG}{a function giving the quasi-random points in \eqn{[0,1]^2} or a name giving
the method to use in the list: \itemize{
 \item \code{sobol}: use of the Sobol sequence
   implemented in \code{randtoolbox::\link[randtoolbox:quasiRNG]{sobol}}
 \item \code{halton}: use of the Halton sequence
   implemented in \code{randtoolbox::\link[randtoolbox:quasiRNG]{halton}}
 \item \code{torus}: use of the Torus sequence
   implemented in \code{randtoolbox::\link[randtoolbox:quasiRNG]{torus}}
}}

\item{ndrawings}{number of replicas of the stochastic estimate of the gradient drawn
at each step. The gradient is computed using the average of these replicas.}
}
\value{
an object of class \code{VineCopula::\link[VineCopula]{BiCop}()}
containing the estimated copula.
}
\description{
This function uses computes the MMD-estimator of a bivariate copula family.
This computation is done through a stochastic gradient algorithm,
that is itself computed by the function \code{\link{BiCopGradMMD}()}.
The main arguments are the two vectors of observations, and the copula family.
The bidimensional copula families are indexed in the same way as
in \code{VineCopula::\link[VineCopula]{BiCop}()} (which computes the MLE estimator).
}
\examples{
# Estimation of a bivariate Gaussian copula with correlation 0.5.
dataSampled = VineCopula::BiCopSim(N = 500, family = 1, par = 0.5)
estimator = BiCopEstMMD(u1 = dataSampled[,1], u2 = dataSampled[,2], family = 1, niter=10)
estimator$par

\donttest{
# Estimation of a bivariate Student copula with correlation 0.5 and 5 degrees of freedom
dataSampled = VineCopula::BiCopSim(N = 1000, family = 2, par = 0.5, par2 = 5)
estimator = BiCopEstMMD(u1 = dataSampled[,1], u2 = dataSampled[,2], family = 2)
estimator$par
estimator$par2


# Comparison with maximum likelihood estimation with and without outliers
dataSampled = VineCopula::BiCopSim(N = 500, family = 1, par = 0.5)
estimatorMMD = BiCopEstMMD(u1 = dataSampled[,1], u2 = dataSampled[,2], family = 1)
estimatorMMD$par
estimatorMLE = VineCopula::BiCopEst(u1 = dataSampled[,1], u2 = dataSampled[,2],
  family = 1, method = "mle")
estimatorMLE$par

dataSampled[1:10,1] = 0.999
dataSampled[1:10,2] = 0.001
estimatorMMD = BiCopEstMMD(u1 = dataSampled[,1], u2 = dataSampled[,2], family = 1)
estimatorMMD$par
estimatorMLE = VineCopula::BiCopEst(u1 = dataSampled[,1], u2 = dataSampled[,2],
  family = 1, method = "mle")
estimatorMLE$par


# Estimation of a bivariate Gaussian copula with real data
data("daxreturns", package = "VineCopula")
BiCopEstMMD(u1 = daxreturns[,1], u2 = daxreturns[,2], family = 1)
estimator$par
}


}
\references{
Alquier, P., Chérief-Abdellatif, B.-E., Derumigny, A., and Fermanian, J.D. (2020).
Estimation of copulas via Maximum Mean Discrepancy.
ArXiv preprint \href{https://arxiv.org/abs/2010.00408}{arxiv:2010.00408}
}
\seealso{
\code{VineCopula::\link[VineCopula]{BiCopEst}()} for other methods of estimation
  such as Maximum Likelihood Estimation or Inversion of Kendall's tau.
\code{\link{BiCopGradMMD}()} for the computation of the stochastic gradient.
\code{\link{BiCopEst.MO}} for the estimation of Marshall-Olkin copulas by MMD.
}
