% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.JointFPM.R
\name{predict.JointFPM}
\alias{predict.JointFPM}
\title{Post-estimation function for JointFPMs}
\usage{
\method{predict}{JointFPM}(object, type = "mean_no", newdata, t, exposed, ci_fit = TRUE, ...)
}
\arguments{
\item{object}{A joint flexible parametric model of class \code{JointFPM}.}

\item{type}{A character vector defining the estimate of interest. Currently available
options are:
\itemize{
\item{\code{mean_no}: }{Estimates the mean number of events at time(s) \code{t}.}
\item{\code{diff}: }{Estimates the difference in mean number of events
between exposed and unexposed at time(s) \code{t}.}
}}

\item{newdata}{A \code{data.frame} including the variable values used for the prediction. One
value for each variable used in either the recurrent or competing event
model is required.}

\item{t}{A vector defining the time points used for the prediction.}

\item{exposed}{A function that takes \code{newdata} as an argument and creates a new dataset
for the exposed group. This argument is required if \code{type = 'diff'}.
Please see details for more information.}

\item{ci_fit}{Logical indicator for whether confidence intervals should be estimated
for the fitted estimates using the delta method.}

\item{...}{Added for compatibility with other predict functions.}
}
\value{
A \code{data.frame} with the following columns:
\itemize{
\item{\code{t}: }{The time for the prediction,}
\item{\code{fit}: }{The point estimate of the prediction,}
\item{\code{lci}: }{The lower confidence interval limit,}
\item{\code{uci}: }{The upper confidence interval limit.}
}
}
\description{
Predicts different estimates from a joint flexible parametric model.
Currently only the estimation of the mean number of events at different
time points is supported.
}
\details{
The function required for the \code{exposed} argument must take the \code{newdata}
dataset as argument and transform it to a new dataset that defines the
exposed group. Assume we assume that we have a model with one variable
\code{trt} which is a 0/1 coded treatment indicator. If we would like to obtain
the difference in mean number of events comparing the untreated to treated
group we could use the following function assuming that
\code{newdata = data.frame(trt = 0)}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{function(x)\{transform(x, trt = 1)\}
}\if{html}{\out{</div>}}
}
\examples{
library(data.table) # For data preparations

# Load bladder cancer dataset from survival package
bldr_df <- as.data.table(survival::bladder1)
bldr_df <- bldr_df[, .(id, treatment, start, stop, status)]

# Define dataset for competing event times
bldr_ce <- bldr_df[, .SD[stop == max(stop)],
                   by = id]

bldr_ce[, `:=`(ce = 1,
               re = 0,
               event = as.numeric(status \%in\% 2:3),
               start = 0)]

# Define dataset for bladder cancer recurrences
bldr_re <- bldr_df[,
                   `:=`(ce = 0,
                        re = 1,
                        event = as.numeric(status == 1))]

# Combine datasets into one stacked dataset

bldr_stacked <- rbindlist(list(bldr_ce, bldr_re))

bldr_stacked[, `:=`(pyridoxine = as.numeric(treatment == "pyridoxine"),
                    thiotepa   = as.numeric(treatment == "thiotepa"))]

bldr_stacked$stop[bldr_stacked$stop == 0] <- 1 # Add one day survival

# Print stacked dataset
head(bldr_stacked)

bldr_model <- JointFPM(Surv(time  = start,
                            time2 = stop,
                            event = event,
                            type  = 'counting') ~ 1,
                       re_model = ~ pyridoxine + thiotepa,
                       ce_model = ~ pyridoxine + thiotepa,
                       re_indicator = "re",
                       ce_indicator = "ce",
                       df_ce = 3,
                       df_re = 3,
                       cluster  = "id",
                       data     = bldr_stacked)

predict(bldr_model,
        newdata = data.frame(pyridoxine = 1,
                             thiotepa   = 0),
        t       =  c(10, 20),
        ci_fit  = FALSE)

}
