% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/6-UserBayesFunctions.R
\name{bayescomp}
\alias{bayescomp}
\title{Bayesian Compound DP-Optimal Designs}
\usage{
bayescomp(formula, predvars, parvars, family = binomial(), prior, alpha, prob,
  lx, ux, iter, k, fimfunc = NULL, ICA.control = list(),
  crt.bayes.control = list(), sens.bayes.control = list(), initial = NULL,
  npar = NULL, plot_3d = c("lattice", "rgl"))
}
\arguments{
\item{formula}{A nonlinear model \code{\link[stats]{formula}}.
A symbolic description of the model consists of predictors and the unknown model parameters.
Will be coerced to a \code{\link[stats]{formula}} if necessary.}

\item{predvars}{A vector of characters. Denotes the predictors in the \code{\link[stats]{formula}}.}

\item{parvars}{A vector of characters. Denotes the unknown parameters in the \code{\link[stats]{formula}}.}

\item{family}{A description of the response distribution and link function to be used in the model.
This can be a family function, a call to a family function or a character string naming the family.
 Every family function has a link argument allowing to specify the link function to be applied on the response variable.
  If not specified, default links are used. For details see \code{\link[stats]{family}}.
   By default, a linear gaussian model \code{gaussian()} is applied.}

\item{prior}{An object of class \code{cprior}. User can also use one of the functions
 \code{\link{uniform}}, \code{\link{normal}},
\code{\link{skewnormal}} or \code{\link{student}}  to create the  prior. See 'Details' of \code{\link{bayes}}.}

\item{alpha}{A value between 0 and 1.
Compound or combined DP-criterion  is the product of the efficiencies of a design  with respect to D- and average P- optimality, weighted by \code{alpha}.}

\item{prob}{Either \code{formula} or a \code{function}. when function, its argument is \code{x} and \code{param} same as the arguments in \code{fimfunc}.
\code{prob} as a function takes the design points and vector of parameters and returns the probability of success at each design points.
See 'Examples'.}

\item{lx}{Vector of lower bounds for the predictors. Should be in the same order as \code{predvars}.}

\item{ux}{Vector of upper bounds for the predictors. Should be in the same order as \code{predvars}.}

\item{iter}{Maximum number of iterations.}

\item{k}{Number of design points. Must be at least equal to the number of model parameters to avoid singularity of the FIM.}

\item{fimfunc}{A function. Returns the FIM as a \code{matrix}. Required when \code{formula} is missing. See 'Details' of \code{\link{minimax}}.}

\item{ICA.control}{ICA control parameters. For details, see \code{\link{ICA.control}}.}

\item{crt.bayes.control}{Control parameters to approximate the integral in  Bayesian criterion at a given design over the parameter space.
For details, see \code{\link{crt.bayes.control}}.}

\item{sens.bayes.control}{Control parameters to verify the general equivalence theorem. For details, see \code{\link{sens.bayes.control}}.}

\item{initial}{A matrix of the  initial designs that will be used as initial solutions (countries).
Every row is a design, i.e.  a concatenation of \code{x} and \code{w}. Will be coerced to a \code{matrix} if necessary.  See 'Details' of \code{\link{minimax}}.}

\item{npar}{Number of model parameters.  Used when \code{fimfunc} is given instead of \code{formula} to specify the number of model parameters.
If not specified truly, the sensitivity (derivative) plot may be shifted below the y-axis. When \code{NULL}, it will be set to \code{length(parvars)} or
\code{prior$npar} when \code{missing(formula)}.}

\item{plot_3d}{Which package should be used to plot the sensitivity (derivative) function for two-dimensional design space. Defaults to \code{"lattice"}.}
}
\value{
an object of class \code{bayes} that is a list including three sub-lists:
\describe{
  \item{\code{arg}}{A list of design and algorithm parameters.}
  \item{\code{evol}}{A list of length equal to the number of iterations that stores the information about the best design (design with least criterion value) of each iteration as follows:
   \code{evol[[iter]]} contains:
    \tabular{lll}{
      \code{iter}                   \tab      \tab Iteration number.\cr
      \code{x}                      \tab      \tab Design points. \cr
      \code{w}                      \tab      \tab Design weights. \cr
      \code{min_cost}               \tab      \tab Cost (Bayesian criterion value) of the best imperialist.  \cr
      \code{mean_cost}              \tab      \tab Mean of costs of all imperialists. \cr
      \code{sens}                   \tab      \tab An object of class 'sensbayes'. See below.\cr
    }
  }

  \item{\code{empires}}{A list of all empires of the last iteration.}
  \item{\code{alg}}{A list with following information:
    \tabular{lll}{
      \code{nfeval}           \tab      \tab Number of function evaluations. See below. \cr
      \code{nlocal}           \tab      \tab Number of successful local search. \cr
      \code{nrevol}           \tab      \tab Number of successful revolutions. \cr
      \code{nimprove}         \tab      \tab Number of successful movements toward the imperialists in assimilation step. \cr
      \code{convergence}      \tab      \tab Stopped by \code{'maxiter'} or \code{'equivalence'}?\cr
    }
  }
}
\code{sens}  contains information about design verification by the general equivalence theorem.
 See \code{sensbayes} for more Details. It is only available every \code{ICA.control$checkfreq} iterations
 and the last iteration if   \code{ICA.control$checkfreq >= 0}. Otherwise, \code{NULL}.

 \code{nfeval} does not count the function evaluations from checking the general equivalence theorem.
}
\description{
Finds compound Bayesian DP-optimal designs that meets the dual goal of the parameter estimation and
  increasing the probability of a particular outcome in a binary response  model.
A compound Bayesian DP-optimal design maximizes  the product of the efficiencies of a design \eqn{\xi} with respect to D- and average P-optimality, weighted by a pre-defined mixing constant
\eqn{0 \leq \alpha \leq 1}{0 <= \alpha <= 1}.
}
\details{
Let \eqn{\Xi} be the space of all  approximate designs with
 \eqn{k} design points (support points) at \eqn{x_1, x_2, ...,  x_k}
  from  design space \eqn{\chi} with
 corresponding weights  \eqn{w_1,... ,w_k}.
 Let \eqn{M(\xi, \theta)} be the Fisher information
  matrix (FIM) of a \eqn{k-}point design \eqn{\xi},
   \eqn{\pi(\theta)} is a user-given  prior distribution for the vector of unknown parameters \eqn{\theta} and
   \eqn{p(x_i, \theta)} is the ith probability of success
given by \eqn{x_i} in a binary response model.
  A Bayesian compound DP-optimal criterion maximizes over \eqn{\Xi}
\deqn{\int_{\theta \in \Theta} \frac{\alpha}{q}\log|M(\xi, \theta)| + (1- \alpha)
\log \left( \sum_{i=1}^k w_ip(x_i, \theta) \right) \pi(\theta) d\theta.}{
integration over \Theta \alpha/q log|M(\xi, \theta)| + (1- \alpha)
log ( \sum w_i p(x_i, \theta)) \pi(\theta) d\theta.
}

Use \code{\link{plot}} function to verify the general equivalence theorem for the output design.

To increase the speed of the algorithm, change the tuning parameters \code{tol} and \code{maxEval} via the
argument \code{crt.bayes.control}.
}
\examples{
##########################################################################
# DP-optimal design for a logitic model with two predictors: with formula
##########################################################################
p <- c(1, -2, 1, -1)
myprior <- uniform(p -1.5, p + 1.5)
myformula1 <- ~exp(b0+b1*x1+b2*x2+b3*x1*x2)/(1+exp(b0+b1*x1+b2*x2+b3*x1*x2))
res1 <- bayescomp(formula = myformula1,
                  predvars = c("x1", "x2"),
                  parvars = c("b0", "b1", "b2", "b3"),
                  family = binomial(),
                  lx = c(-1, -1), ux = c(1, 1),
                  prior = myprior, iter = 1, k = 7,
                  prob = ~1-1/(1+exp(b0 + b1 * x1 + b2 * x2 + b3 * x1 * x2)),
                  alpha = .5, ICA.control = list(rseed = 1366))


\dontrun{
  res1 <- iterate(res1, 1000)
  plot(res1, sens.bayes.control = list(cubature = list(tol = 1e-3, maxEval = 1000)))
}

##########################################################################
# DP-optimal design for a logitic model with two predictors: with fimfunc
##########################################################################
# The function of the Fisher information matrix for this model is 'FIM_logistic_2pred'
# We should reparameterize it to match the standard of the argument 'fimfunc'
myfim <- function(x, w, param){
  npoint <- length(x)/2
  x1 <- x[1:npoint]
  x2 <- x[(npoint+1):(npoint*2)]
  FIM_logistic_2pred(x1 = x1,x2 = x2, w = w, param = param)
}

## The following function is equivalent to the function created
# by the formula: ~1-1/(1+exp(b0 + b1 * x1 + b2 * x2 + b3 * x1 * x2))
# It returns probability of success given x and param
# x = c(x1, x2) and param = c()

myprob <- function(x, param){
  npoint <- length(x)/2
  x1 <- x[1:npoint]
  x2 <- x[(npoint+1):(npoint*2)]
  b0 <- param[1]
  b1 <- param[2]
  b2 <- param[3]
  b3 <- param[4]
  out <- 1-1/(1+exp(b0 + b1 * x1 + b2 * x2 + b3 * x1 * x2))
  return(out)
}
\dontrun{
res2 <- bayescomp(fimfunc = myfim,
                  lx = c(-1, -1), ux = c(1, 1),
                  prior = myprior, iter = 1000, k = 7,
                  prob = myprob, alpha = .5,
                  ICA.control = list(rseed = 1366))
  plot(res2, sens.bayes.control = list(cubature = list(maxEval = 1000, tol = 1e-4)))
}


}
\references{
McGree, J. M., Eccleston, J. A., and Duffull, S. B. (2008). Compound optimal design criteria for nonlinear models. Journal of Biopharmaceutical Statistics, 18(4), 646-661.
}
\seealso{
\code{\link{sensbayescomp}}
}
