% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_GSOD.R
\name{get_GSOD}
\alias{get_GSOD}
\title{Download and Return a Tidy Data Frame of \acronym{GSOD} Weather Data}
\usage{
get_GSOD(years, station = NULL, country = NULL, max_missing = NULL,
  agroclimatology = FALSE)
}
\arguments{
\item{years}{Year(s) of weather data to download.}

\item{station}{Optional. Specify a station or multiple stations for which to
retrieve, check and clean weather data using \var{STNID}. The \acronym{NCEI}
reports years for which the data are available. This function checks against
these years. However, not all cases are properly documented and in some cases
files may not exist on the \acronym{FTP} server even though it is indicated
that data was recorded for the station for a particular year. If a station is
specified that does not have an existing file on the server, this function
will silently fail and move on to existing files for download and cleaning
from the \acronym{FTP} server.}

\item{country}{Optional. Specify a country for which to retrieve weather
data; full name or \acronym{ISO} codes can be used.}

\item{max_missing}{Optional. The maximum number of days allowed to be missing
from a station's data before it is excluded from final file output.}

\item{agroclimatology}{Optional. Logical. Only clean data for stations
between latitudes 60 and -60 for agroclimatology work, defaults to \code{FALSE}.
Set to \code{TRUE} to include only stations within the confines of these
latitudes.}
}
\value{
A data frame as a \code{\link[tibble]{tibble}} object of weather
data.
}
\description{
This function automates downloading, cleaning, reformatting of data from
the Global Surface Summary of the Day (\acronym{GSOD}) data provided by the
\href{https://data.noaa.gov/dataset/dataset/global-surface-summary-of-the-day-gsod/}{US National Centers for Environmental Information (NCEI)},
Three additional useful elements: saturation vapour pressure (es), actual
vapour pressure (ea) and relative humidity (RH) are calculated and returned
in the final data frame.

Parallel processing can be enabled using \code{\link[future]{plan}} to set
up a parallel backend of your choice, e.g.,
\code{future::plan("multisession")}. See examples for more.
}
\details{
Stations reporting a latitude of < -90 or > 90 or longitude of < -180 or >
180 are removed. Stations may be individually checked for number of
missing days to assure data quality and omitting stations with too many
missing observations.

All units are converted to International System of Units (SI), \emph{e.g.}
Fahrenheit to Celsius and inches to millimetres.

Alternative elevation measurements are supplied for missing values or values
found to be questionable based on the Consultative Group for International
Agricultural Research's Consortium for Spatial Information group's
(\acronym{CGIAR-CSI}) Shuttle Radar Topography Mission 90 metre
(\acronym{SRTM} 90m) digital elevation data based on \acronym{NASA}'s
original \acronym{SRTM} 90m data. Further information on these data and
methods can be found on \pkg{GSODR}'s
\href{https://github.com/ropensci/GSODR/blob/master/data-raw/fetch_isd-history.md}{GitHub repository}.

Data summarise each year by station, which include vapour pressure and
relative humidity elements calculated from existing data in \acronym{GSOD}.

All missing values in resulting files are represented as \code{NA} regardless
of which field they occur in.

For a complete list of the fields and description of the contents and units,
please refer to Appendix 1 in the \pkg{GSODR} vignette,
\code{vignette("GSODR", package = "GSODR")}.

For more information see the description of the data provided by
\acronym{NCEI}, \url{http://www7.ncdc.noaa.gov/CDO/GSOD_DESC.txt}.
}
\note{
While \pkg{GSODR} does not distribute GSOD weather data, users of
the data should note the conditions that the U.S. \acronym{NCEI} places upon
the \acronym{GSOD} data.
\dQuote{The following data and products may have conditions placed on
their international commercial use.  They can be used within the U.S. or for
non-commercial international activities without restriction. The non-U.S.
data cannot be redistributed for commercial purposes. Re-distribution of
these data by others must provide this same notification.}
}
\examples{
\donttest{
# Download weather station for Toowoomba, Queensland for 2010
tbar <- get_GSOD(years = 2010, station = "955510-99999")

tbar

# Download data for Australia from 2010 to 2015
AUS <- get_GSOD(years = 2010:2015, country = "Australia")

AUS

# Download agroclimatology data for 2015 using parallel processing
future::plan("multisession")
ag <- get_GSOD(years = 2015, agroclimatology = TRUE)

ag

# Download global data for 2010 to 2015 with a maximum allowed 5 missing days
# of data using parallel processing

future::plan("multisession")
global <- get_GSOD(years = 2010:2015, max_missing = 5)

global

}
}
\references{
{Jarvis, A., Reuter, H. I, Nelson, A., Guevara, E. (2008)
Hole-filled SRTM for the globe Version 4, available from the CGIAR-CSI SRTM
90m Database \url{http://srtm.csi.cgiar.org}}
}
\seealso{
\code{\link{reformat_GSOD}}
}
\author{
Adam H Sparks, \email{adamhsparks@gmail.com}
}
