% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crit_SMS.R
\name{crit_SMS}
\alias{crit_SMS}
\title{Analytical expression of the SMS-EGO criterion with m>1 objectives}
\usage{
crit_SMS(x, model, paretoFront = NULL, critcontrol = list(epsilon = 1e-06,
  gain = 1), type = "UK")
}
\arguments{
\item{x}{a vector representing the input for which one wishes to calculate the criterion,}

\item{model}{a list of objects of class \code{\link[DiceKriging]{km}} (one for each objective),}

\item{paretoFront}{(optional) matrix corresponding to the Pareto front of size \code{[n.pareto x n.obj]},}

\item{critcontrol}{list with arguments: 
\itemize{
\item \code{currentHV} current hypervolume;
\item \code{refPoint} reference point for hypervolume computations;
\item \code{epsilon} optional value to use in additive epsilon dominance;
\item \code{gain} optional gain factor for sigma.
}
       Options for the \code{\link[GPareto]{checkPredict}} function: \code{threshold} (\code{1e-4}) and \code{distance} (\code{covdist}) are used to avoid numerical issues occuring when adding points too close to the existing ones.}

\item{type}{"\code{SK}" or "\code{UK}" (by default), depending whether uncertainty related to trend 
estimation has to be taken into account.}
}
\value{
Value of the criterion.
}
\description{
Computes a slightly modified infill Criterion of the SMS-EGO.
To avoid numerical instabilities, an additional penalty is added to the new point if it is too close to an existing observation.
}
\examples{
#---------------------------------------------------------------------------
# SMS-EGO surface associated with the "P1" problem at a 15 points design
#---------------------------------------------------------------------------
set.seed(25468)
library(DiceDesign)

n_var <- 2 
f_name <- "P1" 
n.grid <- 26
test.grid <- expand.grid(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid))
n_appr <- 15 
design.grid <- round(maximinESE_LHS(lhsDesign(n_appr, n_var, seed = 42)$design)$design, 1)
response.grid <- t(apply(design.grid, 1, f_name))
PF <- t(nondominated_points(t(response.grid)))
mf1 <- km(~., design = design.grid, response = response.grid[,1])
mf2 <- km(~., design = design.grid, response = response.grid[,2])

model <- list(mf1, mf2)
critcontrol <- list(refPoint = c(300, 0), currentHV = dominated_hypervolume(t(PF), c(300, 0)))
SMSEGO_grid <- apply(test.grid, 1, crit_SMS, model = model,
                     paretoFront = PF, critcontrol = critcontrol)

filled.contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid),
               matrix(pmax(0, SMSEGO_grid), nrow = n.grid), nlevels = 50,
               main = "SMS-EGO criterion (positive part)", xlab = expression(x[1]),
               ylab = expression(x[2]), color = terrain.colors,
               plot.axes = {axis(1); axis(2);
                            points(design.grid[,1],design.grid[,2], pch = 21, bg = "white")
                            }
              )
}
\references{
W. Ponweiser, T. Wagner, D. Biermann, M. Vincze (2008), Multiobjective Optimization on a Limited Budget of Evaluations Using Model-Assisted S-Metric Selection,
\emph{Parallel Problem Solving from Nature}, pp. 784-794. Springer, Berlin. \cr \cr
T. Wagner, M. Emmerich, A. Deutz, W. Ponweiser (2010), On expected-improvement criteria for model-based multi-objective optimization.   
\emph{Parallel Problem Solving from Nature}, pp. 718-727. Springer, Berlin.
}
\seealso{
\code{\link[GPareto]{crit_EHI}}, \code{\link[GPareto]{crit_SUR}}, \code{\link[GPareto]{crit_EMI}}.
}

