% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cov_cs.R
\name{cscov}
\alias{cscov}
\alias{cscov.default}
\alias{cscov.ols}
\alias{cscov.str}
\alias{cscov.wls}
\alias{cscov.shr}
\alias{cscov.sam}
\title{Cross-sectional covariance matrix approximation}
\usage{
cscov(comb = "ols", n = NULL, agg_mat = NULL, res, mse = TRUE,
      shrink_fun = shrink_estim, ...)
}
\arguments{
\item{comb}{A string specifying the reconciliation method.
\itemize{
\item Ordinary least squares:
\itemize{
\item "\code{ols}" (\emph{default}) - identity error covariance matrix.
}
\item Weighted least squares:
\itemize{
\item "\code{str}" - structural variances.
\item "\code{wls}" - series variances (uses \code{res}).
}
\item Generalized least squares:
\itemize{
\item "\code{shr}"/"\code{sam}" - shrunk/sample covariance (uses \code{res}).
}
}}

\item{n}{Number of variables (\eqn{n = n_a + n_b}).}

\item{agg_mat}{A (\eqn{n_a \times n_b}) numeric matrix representing the cross-sectional
aggregation matrix. It maps the \eqn{n_b} bottom-level (free)
variables into the \eqn{n_a} upper (constrained) variables.}

\item{res}{An (\eqn{N \times n}) optional numeric matrix containing the in-sample
residuals. This matrix is used to compute some covariance matrices.}

\item{mse}{If \code{TRUE} (\emph{default}) the residuals used to compute the covariance
matrix are not mean-corrected.}

\item{shrink_fun}{Shrinkage function of the covariance matrix, \link{shrink_estim} (\emph{default}).}

\item{...}{Not used.}
}
\value{
A (\eqn{n \times n}) symmetric positive (semi-)definite matrix.
}
\description{
This function provides an approximation of the cross-sectional base forecasts errors
covariance matrix using different reconciliation methods (see Wickramasuriya et al.,
2019 and Di Fonzo and Girolimetto, 2023).
}
\examples{
# Aggregation matrix for Z = X + Y
A <- t(c(1,1))
# (10 x 3) in-sample residuals matrix (simulated)
res <- t(matrix(rnorm(n = 30), nrow = 3))

cov1 <- cscov("ols", n = 3)          # OLS methods
cov2 <- cscov("str", agg_mat = A)   # STR methods
cov3 <- cscov("wls", res = res)      # WLS methods
cov4 <- cscov("shr", res = res)      # SHR methods
cov5 <- cscov("sam", res = res)      # SAM methods

# Custom covariance matrix
cscov.ols2 <- function(comb, x) diag(x)
cscov(comb = "ols2", x = 3) # == cscov("ols", n = 3)

}
\references{
Di Fonzo, T. and Girolimetto, D. (2023), Cross-temporal forecast reconciliation:
Optimal combination method and heuristic alternatives, \emph{International Journal
of Forecasting}, 39, 1, 39-57. \doi{10.1016/j.ijforecast.2021.08.004}

Wickramasuriya, S.L., Athanasopoulos, G. and Hyndman, R.J. (2019), Optimal forecast
reconciliation for hierarchical and grouped time series through trace minimization,
\emph{Journal of the American Statistical Association}, 114, 526, 804-819.
\doi{10.1080/01621459.2018.1448825}
}
\seealso{
Cross-sectional framework: 
\code{\link{csboot}()},
\code{\link{csbu}()},
\code{\link{cslcc}()},
\code{\link{csmo}()},
\code{\link{csrec}()},
\code{\link{cstd}()},
\code{\link{cstools}()}
}
\concept{Framework: cross-sectional}
