% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distance_to_treeline.R
\name{distance_to_treeline}
\alias{distance_to_treeline}
\title{Wrapper that calculates the distance relative to the nearest local treeline}
\usage{
distance_to_treeline(lon, lat, gstRaster, gslRaster, elevationRaster, elevation,
                            pointDf , gridSize = 10,  gridStepSize = 0.0025, plot = FALSE,
                            plotZoom = NULL, treelineSamplingSize = 10, plotHist = FALSE,
                            gstMin = 6.4, gslMin = 94)
}
\arguments{
\item{lon}{Longitude of a point (in degrees; WGS 84). One value or a vector, data type "numeric" and finite.}

\item{lat}{Latitude of a point (in degrees; WGS 84). One value or a vector, data type "numeric" and finite.}

\item{gstRaster}{Climatic raster that contains the growing season temperature. Data type "SpatRaster".}

\item{gslRaster}{Climatic raster that contains the growing season length. Data type "SpatRaster".}

\item{elevationRaster}{Raster that contains a digital elevation model. Data type "SpatRaster".}

\item{elevation}{Elevation of the point of interest (in meters above the sea level). One value or a vector, data type "numeric" and finite.}

\item{pointDf}{Data frame that contains coordinates (WGS 84) of points above the treeline.
The first column must contain the longitude, the second the latitude.
The values must be of the data type "numeric" and finite.}

\item{gridSize}{Square size (in km) of the grid. One value, data type "numeric" and finite.}

\item{gridStepSize}{Step size for the square sampling (in degree) of the grid. One value, data type "numeric" and finite.}

\item{plot}{Boolean that defines if a map of the sampled area is plotted. The plot will only be shown if the value is \code{TRUE}.}

\item{plotZoom}{Map zoom, for the "get_map" function of the "ggmap" library. One value, data type "integer", from 3 to 21 and finite.}

\item{treelineSamplingSize}{A constant number of samples taken from one "treeline piece". One value, data type "integer", not zero and finite.}

\item{plotHist}{Boolean that defines if a histogram of the sampled elevation is plotted. The plot will only be shown if the value is \code{TRUE}.}

\item{gstMin}{Growing season temperature threshold for tree growth (in degree Celsius). One value, data type "numeric" and finite.}

\item{gslMin}{Growing season length threshold for tree growth (days). One value, data type "numeric" and finite.}
}
\value{
Returns the distance to the local treeline in meters as one value or as vector.
}
\description{
Calculate the distance to the treeline in meters. Positive values indicate that the sample is above the treeline.
Negative values for samples below the treeline.
}
\details{
This is the main function, which calls the other relevant functions. Specifically, in turn, it calls \code{get_nearest_point} to
identify where the nearest local treeline is, \code{generate_grid}, \code{classify_above_treeline}, and \code{sample_treeline} to locally
investigate at what elevation the treeline is, and finally \code{calculate_distance} to determine the elevation of the point relative to
the local treeline. It is recommended to use this wrapper rather than the individual functions, unless you have a very specific reason not to.
The position of a point relative to the treeline depends on a treeline definition.  Here we follow the definition of Paulsen & Körner,
Alp. Bot. 124: 1-12 (2014), which is based on specific thresholds of growing season length and growing season temperature (94 days and 9.4°C,
respectively). It is possible to adjust these thresholds manually, e.g. to achieve a elevation above or below another climatic line.
Note that this requires to first calculate \code{pointDf} for the boundary of interest using the functions \code{generate_point_df}.
Because the implemented treeline definition depends not only on temperature but also on growing season length, it can be affected by drought.
Therefore, the user must take care in interpreting treeline information in desert mountain systems. Here, we recommend to frequently use
the option \code{plot} and \code{plotHist} to gain a thorough understanding of the local situation.
}
\examples{
#Get raster layer from CHELSA
gstURL <- paste0("https://os.zhdk.cloud.switch.ch/chelsav2/GLOBAL/",
                 "climatologies/1981-2010/bio/CHELSA_gst_1981-2010_V.2.1.tif")
gslURL <- paste0("https://os.zhdk.cloud.switch.ch/chelsav2/GLOBAL/",
                 "climatologies/1981-2010/bio/CHELSA_gsl_1981-2010_V.2.1.tif")
\donttest{
gst <- terra::rast(gstURL, vsi = TRUE)
gsl <- terra::rast(gslURL, vsi = TRUE)

gmted2010URL <- paste0("https://edcintl.cr.usgs.gov/downloads/sciweb1/shared/topo/downloads/GMTED/",
                      "Global_tiles_GMTED/300darcsec/med/E000/30N000E_20101117_gmted_med300.tif")
gmted2010Part <- terra::rast(gmted2010URL, vsi = TRUE)

#Check one point
distance_to_treeline(lon = 8.65, lat = 46.87, gstRaster = gst, gslRaster = gsl,
                     elevationRaster = gmted2010Part, elevation = 504,
                     pointDf = pointsAboveTreeline, plot = FALSE,
                     plotHist = FALSE, gstMin = 6.4, gslMin = 94)
distance_to_treeline(lon = 4.47, lat = 51.92, gstRaster = gst, gslRaster = gsl,
                     elevationRaster = gmted2010Part, elevation = 504,
                     pointDf = pointsAboveTreeline, plot = FALSE,
                     plotHist = FALSE, gstMin = 6.4, gslMin = 94)
distance_to_treeline(lon = -156.71, lat = 69.74,gstRaster = gst, gslRaster = gsl,
                     elevationRaster = gmted2010Part, elevation = 504,
                     pointDf = pointsAboveTreeline, plot = FALSE,
                     plotHist = FALSE, gstMin = 6.4, gslMin = 94)
}
}
\author{
Livio Bätscher, Jurriaan M. de Vos
}
