% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kernel_fun.R
\name{kernel}
\alias{kernel}
\alias{kernel.DBH.fast}
\alias{kernel.DBH.crit}
\alias{kernel.ADBH.fast}
\alias{kernel.ADBH.crit}
\title{Kernel functions for discrete.BH}
\usage{
kernel.DBH.fast(stepf, pv.numer, pv.denom)

kernel.DBH.crit(stepf, pv.numer, pv.denom, alpha, sorted.pv)

kernel.ADBH.fast(stepf, pv.numer, pv.denom)

kernel.ADBH.crit(stepf, pv.numer, pv.denom, alpha, sorted.pv)
}
\arguments{
\item{stepf}{a list of the CDFs under the null hypothesis of each p-value.}

\item{pv.numer}{a numeric vector. Contains all values of the p-values supports if we search for the critical constants. If not, contains only the observed p-values. Must be in increasing order.}

\item{pv.denom}{a numeric vector. Identical to \code{pv.numer} for a step-down procedure. Equals \code{c.m} for a step-up procedure.}

\item{alpha}{the target FDR level, a number strictly between 0 and 1.}

\item{sorted.pv}{a vector of observed p-values, in increasing order.}
}
\value{
For \code{kernel.DBH.fast} and \code{kernel.ADBH.fast}, a vector of transformed p-values is returned.
\code{kernel.DBH.crit} and \code{kernel.ADBH.crit} return a list object with critical constants (\code{$crit.consts})
and transformed p-values (\code{$pval.transf}), but only if \code{pv.numer = pv.denom}.
}
\description{
Kernel functions that transform 
observed p-values or their support according to [HSU], [HSD],
[AHSU] and [AHSD]. 
The output is used by \code{\link{discrete.BH}}.
Additionally, \code{kernel.DBH.crit} and \code{kernel.ADBH.crit}
compute and return the critical constants.
The end user should not use them.
}
\details{
When computing critical constants under step-down, that is,
when using \code{kernel.DBH.crit} or \code{kernel.ADBH.crit}
with \code{pv.numer = pv.denom} (i.e. the step-down case),
we still need to get transformed p-values to compute
the adjusted p-values.

This version: 2018-11-13.
}
\examples{
X1 <- c(4, 2, 2, 14, 6, 9, 4, 0, 1)
X2 <- c(0, 0, 1, 3, 2, 1, 2, 2, 2)
N1 <- rep(148, 9)
N2 <- rep(132, 9)
Y1 <- N1-X1
Y2 <- N2-X2
df <- data.frame(X1,Y1,X2,Y2)
df
#Construction of the p-values and their support
df.formatted <- fisher.pvalues.support(counts = df, input = "noassoc")
raw.pvalues <- df.formatted$raw
pCDFlist <- df.formatted$support

m <- length(raw.pvalues)
alpha <- 0.05

# Compute the step functions from the supports
stepf <- build.stepfuns(pCDFlist)

# We stay in a step-down context, where pv.numer = pv.denom,
# for the sake of simplicity

# If not searching for critical constants, we use only the observed p-values
sorted.pvals <- sort(raw.pvalues)
y.DBH.fast <- kernel.DBH.fast(stepf, sorted.pvals, sorted.pvals)
y.ADBH.fast <- kernel.ADBH.fast(stepf, sorted.pvals, sorted.pvals)

# If searching for critical constants, we use (almost) the complete support
pv.list.all <- sort(unique(as.numeric(unlist(pCDFlist))))
# apply the shortcut drawn from Lemma 3, that is
# c.1 >= the effective critical value associated to (alpha/m)/(1 + alpha/m)
pv.list <- short.eff(pv.list.all, alpha/(m + alpha))
# then re-add the observed p-values (needed to compute the adjusted p-values),
# because we may have removed some of them by the shortcut
pv.list <- sort(unique(c(pv.list, sorted.pvals)))
# compute transformed support
y.DBH.crit <- kernel.DBH.crit(stepf, pv.list, pv.list, alpha, sorted.pvals)
y.ADBH.crit <- kernel.ADBH.crit(stepf, pv.list, pv.list, alpha, sorted.pvals)
# critical constants
y.DBH.crit$crit.consts
y.ADBH.crit$crit.consts
# The following exist only for step-down direction
y.DBH.crit$pval.transf
y.ADBH.crit$pval.transf

}
\seealso{
\code{\link{discrete.BH}}, \code{\link{DiscreteFDR}},
\code{\link{DBR}}, \code{\link{kernel.DBR}}
}
