% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/powerBuyseTest.R
\name{powerBuyseTest}
\alias{powerBuyseTest}
\title{Performing simulation studies with BuyseTest}
\usage{
powerBuyseTest(
  sim,
  sample.size,
  n.rep = c(1000, 10),
  null = c(netBenefit = 0),
  cpus = 1,
  export.cpus = NULL,
  seed = NULL,
  conf.level = NULL,
  power = NULL,
  max.sample.size = 2000,
  alternative = NULL,
  order.Hprojection = NULL,
  transformation = NULL,
  trace = 1,
  ...
)
}
\arguments{
\item{sim}{[function] take two arguments:
the sample size in the control group (\code{n.C}) and the sample size in the treatment group (\code{n.C})
and generate datasets. The datasets must be data.frame objects or inherits from data.frame.}

\item{sample.size}{[integer vector or matrix, >0] the group specific sample sizes relative to which the simulations should be perform.
When a vector, the same sample size is used for each group. Alternatively can be a matrix with two columns, one for each group (respectively T and C).}

\item{n.rep}{[integer, >0] the number of simulations.
When specifying the power instead of the sample size, should be a vector of length 2 where the second element indicates the number of simulations used to identify the sample size.}

\item{null}{[numeric vector] For each statistic of interest, the null hypothesis to be tested.
The vector should be named with the names of the statistics.}

\item{cpus}{[integer, >0] the number of CPU to use. Default value is 1.}

\item{export.cpus}{[character vector] name of the variables to export to each cluster.}

\item{seed}{[integer, >0] Random number generator (RNG) state used when starting the simulation study.
If \code{NULL} no state is set.}

\item{conf.level}{[numeric, 0-1] type 1 error level.
Default value read from \code{BuyseTest.options()}.}

\item{power}{[numeric, 0-1] type 2 error level used to determine the sample size. Only relevant when \code{sample.size} is not given. See details.}

\item{max.sample.size}{[interger, 0-1] sample size used to approximate the sample size achieving the requested type 1 and type 2 error (see details).
Can have length 2 to indicate the sample in each group (respectively T and C) when the groups have unequal sample size.}

\item{alternative}{[character] the type of alternative hypothesis: \code{"two.sided"}, \code{"greater"}, or \code{"less"}.
Default value read from \code{BuyseTest.options()}.}

\item{order.Hprojection}{[integer 1,2] the order of the H-project to be used to compute the variance of the net benefit/win ratio.
Default value read from \code{BuyseTest.options()}.}

\item{transformation}{[logical] should the CI be computed on the logit scale / log scale for the net benefit / win ratio and backtransformed.
Otherwise they are computed without any transformation.
Default value read from \code{BuyseTest.options()}.}

\item{trace}{[integer] should the execution of the function be traced?}

\item{...}{other arguments (e.g. \code{scoring.rule}, \code{method.inference}) to be passed to \code{initializeArgs}.}
}
\value{
An S4 object of class  \code{\linkS4class{S4BuysePower}}.
}
\description{
Performs a simulation studies for several sample sizes.
Returns estimates, their standard deviation, the average estimated standard error, and the rejection rate.
Can also be use for power calculation or to approximate the sample size needed to reach a specific power.
}
\details{
\bold{Sample size calculation}: to approximate the sample size achieving the requested type 1 (\eqn{\alpha}) and type 2 error (\eqn{\beta}),
GPC are applied on a large sample (as defined by the argument \code{max.sample.size}): \eqn{N^*=m^*+n^*} where \eqn{m^*} is the sample size in the control group and \eqn{n^*} is the sample size in the active group.
Then the effect (\eqn{\delta}) and the asymptotic variance of the estimator (\eqn{\sigma^2}) are estimated. The total sample size is then deduced as (two-sided case):
\deqn{\hat{N} = \hat{\sigma}^2\frac{(u_{1-\alpha/2}+u_{1-\beta})^2}{\hat{\delta}^2}} from which the group specific sample sizes are deduced: \eqn{\hat{m}=\hat{N}\frac{m^*}{N^*}} and \eqn{\hat{n}=\hat{N}\frac{n^*}{N^*}}. Here \eqn{u_x} denotes the x-quantile of the normal distribution. \cr
This approximation can be improved by increasing the sample size (argument \code{max.sample.size}) and/or by performing it multiple times based on a different dataset and average estimated sample size per group (second element of argument \code{n.rep}). \cr
To evaluate the approximation, a simulation study is then performed with the estimated sample size. It will not exactly match the requested power but should provide a reasonnable guess which can be refined with further simulation studies. The larger the sample size (and/or number of CPUs) the more accurate the approximation.

\bold{seed}: the seed is used to generate one seed per simulation. These simulation seeds are the same whether one or several CPUs are used.
}
\examples{
library(data.table)

#### Using simBuyseTest ####
## save time by not generating TTE outcomes
simBuyseTest2 <- function(...){simBuyseTest(..., argsCont = NULL, argsTTE = NULL)}

## only point estimate
\dontrun{
pBT <- powerBuyseTest(sim = simBuyseTest2, sample.size = c(10, 25, 50, 75, 100), 
                  formula = treatment ~ bin(toxicity), seed = 10, n.rep = 1000,
                  method.inference = "none", keep.pairScore = FALSE, cpus = 5)
summary(pBT)
model.tables(pBT)
}

## point estimate with rejection rate
\dontshow{
powerBuyseTest(sim = simBuyseTest2, sample.size = c(10, 50, 100), 
               formula = treatment ~ bin(toxicity), seed = 10, n.rep = 10,
               method.inference = "u-statistic", trace = 4)
}
\dontrun{
powerBuyseTest(sim = simBuyseTest2, sample.size = c(10, 50, 100), 
               formula = treatment ~ bin(toxicity), seed = 10, n.rep = 1000,
               method.inference = "u-statistic", trace = 4)
}

#### Using user defined simulation function ####
## power calculation for Wilcoxon test
simFCT <- function(n.C, n.T){
    out <- rbind(cbind(Y=stats::rt(n.C, df = 5), group=0),
                 cbind(Y=stats::rt(n.T, df = 5), group=1) + 1)
    return(data.table::as.data.table(out))
}
simFCT2 <- function(n.C, n.T){
    out <- rbind(cbind(Y=stats::rt(n.C, df = 5), group=0),
                 cbind(Y=stats::rt(n.T, df = 5), group=1) + 0.25)
    return(data.table::as.data.table(out))
}

\dontshow{
powerW <- powerBuyseTest(sim = simFCT, sample.size = c(5, 10,20,30,50,100),
                         n.rep = 10, formula = group ~ cont(Y))
summary(powerW)
}
\dontrun{
powerW <- powerBuyseTest(sim = simFCT, sample.size = c(5,10,20,30,50,100),
                         n.rep = 1000, formula = group ~ cont(Y), cpus = "all")
summary(powerW)
} 

## sample size needed to reach (approximately) a power
## based on summary statistics obtained on a large sample 
\dontrun{
sampleW <- powerBuyseTest(sim = simFCT, power = 0.8, formula = group ~ cont(Y), 
                         n.rep = c(1000,10), max.sample.size = 2000, cpus = 5,
                         seed = 10)
nobs(sampleW)
summary(sampleW) ## not very accurate but gives an order of magnitude

sampleW2 <- powerBuyseTest(sim = simFCT2, power = 0.8, formula = group ~ cont(Y), 
                         n.rep = c(1000,10), max.sample.size = 2000, cpus = 5,
                         seed = 10)
summary(sampleW2) ## more accurate when the sample size needed is not too small
}

}
\author{
Brice Ozenne
}
\keyword{htest}
