% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{bdblockSum}
\alias{bdblockSum}
\title{Block-Based Matrix Addition}
\usage{
bdblockSum(
  A,
  B,
  block_size = NULL,
  paral = NULL,
  byBlocks = TRUE,
  threads = NULL
)
}
\arguments{
\item{A}{Matrix or vector. First input operand.}

\item{B}{Matrix or vector. Second input operand.}

\item{block_size}{Integer. Block size for computation. If NULL, uses maximum
allowed block size.}

\item{paral}{Logical. If TRUE, enables parallel computation. Default is FALSE.}

\item{byBlocks}{Logical. If TRUE (default), forces block-based computation for
large matrices. Can be set to FALSE to disable blocking.}

\item{threads}{Integer. Number of threads for parallel computation. If NULL,
uses half of available threads.}
}
\value{
Matrix or vector containing the result of A + B.
}
\description{
Performs efficient matrix addition using block-based algorithms. The function
supports various input combinations (matrix-matrix, matrix-vector, vector-vector)
and provides options for parallel processing and block-based computation.
}
\details{
This function implements block-based matrix addition algorithms optimized
for cache efficiency and memory usage. Key features:
\itemize{
\item Input combinations supported:
\itemize{
\item Matrix-matrix addition
\item Matrix-vector addition (both left and right)
\item Vector-vector addition
}
\item Performance optimizations:
\itemize{
\item Block-based computation for cache efficiency
\item Parallel processing for large matrices
\item Automatic method selection based on input size
\item Memory-efficient implementation
}
}

The function automatically selects the appropriate addition method based
on input types and sizes. For large matrices (>2.25e+08 elements), block-based
computation is used by default.
}
\examples{
\dontrun{
library(BigDataStatMeth)

# Matrix-matrix addition
N <- 2500
M <- 400
nc <- 4

set.seed(555)
mat1 <- matrix(rnorm(N*M, mean=0, sd=10), N, M)
mat2 <- matrix(rnorm(N*M, mean=0, sd=10), N, M)

# Parallel block addition
result <- bdblockSum(mat1, mat2,
                     paral = TRUE,
                     threads = nc)

# Matrix-vector addition
vec <- rnorm(M)
result_mv <- bdblockSum(mat1, vec,
                        paral = TRUE,
                        threads = nc)
}

}
\references{
\itemize{
\item Golub, G. H., & Van Loan, C. F. (2013). Matrix Computations, 4th Edition.
Johns Hopkins University Press.
\item Kumar, V. et al. (1994). Introduction to Parallel Computing: Design and
Analysis of Algorithms. Benjamin/Cummings Publishing Company.
}
}
\seealso{
\itemize{
\item \code{\link{bdblockSubstract}} for block-based matrix subtraction
\item \code{\link{bdblockMult}} for block-based matrix multiplication
}
}
