% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{bdDiag_subtract_hdf5}
\alias{bdDiag_subtract_hdf5}
\title{Subtract Diagonal Elements from HDF5 Matrices or Vectors}
\usage{
bdDiag_subtract_hdf5(
  filename,
  group,
  A,
  B,
  groupB = NULL,
  target = NULL,
  outgroup = NULL,
  outdataset = NULL,
  paral = NULL,
  threads = NULL,
  overwrite = NULL
)
}
\arguments{
\item{filename}{String. Path to the HDF5 file containing the datasets.}

\item{group}{String. Group path containing the first dataset (A, minuend).}

\item{A}{String. Name of the first dataset (minuend).}

\item{B}{String. Name of the second dataset (subtrahend).}

\item{groupB}{Optional string. Group path containing dataset B.
If NULL, uses same group as A.}

\item{target}{Optional string. Where to write result: "A", "B", or "new" (default: "new").}

\item{outgroup}{Optional string. Output group path.
Default is "OUTPUT".}

\item{outdataset}{Optional string. Output dataset name.
Default is "A_-_B" with .diag suffix if appropriate.}

\item{paral}{Optional logical. Whether to use parallel processing.
Default is FALSE.}

\item{threads}{Optional integer. Number of threads for parallel processing.
If NULL, uses maximum available threads.}

\item{overwrite}{Optional logical. Whether to overwrite existing datasets.
Default is FALSE.}
}
\value{
List with components:
\describe{
\item{fn}{Character string with the HDF5 filename}
\item{ds}{Character string with the full dataset path to the diagonal subtraction result (group/dataset)}
}
}
\description{
Performs optimized diagonal subtraction between two datasets stored in HDF5 format.
Automatically detects whether inputs are matrices (extracts diagonals) or vectors
(direct operation) and uses the most efficient approach. This function is ~50-250x
faster than traditional matrix operations for diagonal computations.
}
\details{
This function provides flexible diagonal subtraction with automatic optimization:
\itemize{
\item Operation modes:
\itemize{
\item Matrix - Matrix: Extract diagonals → vector subtraction → save as vector
\item Matrix - Vector: Extract diagonal → vector subtraction → save as vector
\item Vector - Vector: Direct vector subtraction (most efficient)
}
\item Performance features:
\itemize{
\item Uses optimized vector operations for maximum efficiency
\item Automatic type detection and dimension validation
\item Memory-efficient processing for large datasets
\item Parallel processing support for improved performance
}
\item Validation checks:
\itemize{
\item Matrix inputs must be square (N×N)
\item Vector inputs must have compatible dimensions
\item Automatic dimension matching between operands
}
}
}
\examples{
\dontrun{
library(BigDataStatMeth)

# Create test matrices
N <- 1000
set.seed(123)
A <- matrix(rnorm(N*N), N, N)
B <- matrix(rnorm(N*N), N, N)

# Save to HDF5
bdCreate_hdf5_matrix("test.hdf5", A, "data", "matrixA",
                     overwriteFile = TRUE)
bdCreate_hdf5_matrix("test.hdf5", B, "data", "matrixB",
                     overwriteFile = FALSE)

# Subtract diagonals
result <- bdDiag_subtract_hdf5("test.hdf5", "data", "matrixA", "matrixB",
                              outgroup = "results",
                              outdataset = "diagonal_diff",
                              paral = TRUE)
}

}
