% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bcov.R
\name{bcov.test}
\alias{bcov.test}
\alias{bcov.test.default}
\alias{bcov.test.formula}
\title{Ball Covariance Test}
\usage{
bcov.test(x, ...)

\method{bcov.test}{default}(x, y = NULL, num.permutations = 99,
  distance = FALSE, weight = FALSE, seed = 4, num.threads = 1, ...)

\method{bcov.test}{formula}(formula, data, subset, na.action, ...)
}
\arguments{
\item{x}{a numeric vector, matirx, data.frame or \code{dist} object or list containing numeric vector, matrix, data.frame, or \code{dist} object.}

\item{...}{further arguments to be passed to or from methods.}

\item{y}{a numeric vector, matirx, data.frame or \code{dist} object.}

\item{num.permutations}{the number of permutation replications, when \code{num.permutations} equals to 0, the function returns
the sample version of ball divergence. Default: \code{num.permutations = 99}}

\item{distance}{if \code{distance = TRUE}, \code{x} and \code{y} will be considered as a distance matrix. Default: \code{distance = FALSE}}

\item{weight}{a logical or character value used to choose the form of weight. 
If \code{weight = FALSE}, the ball covariance / correlation with constant weight is used. 
Alternatively, \code{weight = TRUE} and \code{weight = "prob"} indicates the probability weight is chosen 
while setting \code{weight = "chisq"} means select the Chi-square weight. 
Note that this arguments actually only influences the printed result in R console and is 
only available for the \code{bcov.test} function at present. Default: \code{weight = FALSE}}

\item{seed}{the random seed.}

\item{num.threads}{Number of threads. Default \code{num.threads = 1}.}

\item{formula}{a formula of the form \code{~ u + v}, where each of \code{u} and \code{v} are numeric variables giving the data values for one sample. The samples must be of the same length.}

\item{data}{an optional matrix or data frame (or similar: see \code{model.frame}) containing the variables in the formula formula. By default the variables are taken from environment(formula).}

\item{subset}{an optional vector specifying a subset of observations to be used.}

\item{na.action}{a function which indicates what should happen when the data contain \code{NA}s. Defaults to \code{getOption("na.action")}.}
}
\value{
bcov.test returns a list with class "htest" containing the following components:
\item{\code{statistic}}{ball covariance or ball correlation statistic.}            
\item{\code{p.value}}{the p-value for the test.}  
\item{\code{replicates}}{permutation replications of the test statistic.}
\item{\code{size}}{sample size.} 
\item{\code{complete.info}}{a \code{list} containing multiple statistics value and their corresponding $p$ value.}
\item{\code{alternative}}{a character string describing the alternative hypothesis.}
\item{\code{method}}{a character string indicating what type of test was performed.}
\item{\code{data.name}}{description of data.}
}
\description{
Ball covariance test of multivariate independence. 
Ball covariance are generic multivariate measures of dependence in Banach space.
}
\details{
\code{bcov.test} are non-parametric tests of multivariate independence in Banach space. 
The test decision is obtained via permutation, with \code{num.permutations} replicates.

If two samples are pass to arguments \code{x} and \code{y}, the sample sizes (i.e. number of rows or length of the vector) 
of the two variables must agree. If a \code{\link{list}} object is passed to \code{x}, 
each element must with same sample sizes. Moreover, data pass to \code{x} or \code{y} 
must not contain missing or infinite values. 
If we set \code{distance = TRUE}, arguments \code{x}, \code{y} can be a \code{dist} object or a
symmetric numeric matrix recording distance between samples; 
otherwise, these arguments are treated as data.

The \code{bcov.test} statistic is \code{bcov} or \code{ bcor} which are dependence measure 
in Banach space. The \code{bcor} test statistic is based on the normalized 
coefficient of ball covariance. (See the manual page for \code{\link{bcov}} or \code{\link{bcor}}.)

For the general problem of testing independence when the distributions of \eqn{X} and 
\eqn{Y} are unknown, the test based on \code{bcov} can be implemented as a permutation test.
See (Jin et al 2018) for theoretical properties of the test, including statistical consistency.
}
\examples{
set.seed(1)

################# Quick Start #################
error <- runif(50, min = -0.3, max = 0.3)
x <- runif(50, 0, 4*pi)
y <- cos(x) + error
# plot(x, y)
bcov.test(x, y)

################# Quick Start #################
x <- matrix(runif(50 * 2, -pi, pi), nrow = 50, ncol = 2)
error <- runif(50, min = -0.3, max = 0.3)
y <- (sin((x[,1])^2 + x[,2])) + error
bcov.test(x = x, y = y)

################# Ball Covariance Test for Non-Hilbert Data #################
# load data:
data("ArcticLake")
# Distance matrix between y:
Dy <- nhdist(ArcticLake[["x"]], method = "compositional")
# Distance matrix between x:
Dx <- dist(ArcticLake[["depth"]])
# hypothesis test with BCov:
bcov.test(x = Dx, y = Dy, distance = TRUE)

################  Weighted Ball Covariance Test  #################
data("ArcticLake")
Dy <- nhdist(ArcticLake[["x"]], method = "compositional")
Dx <- dist(ArcticLake[["depth"]])
# hypothesis test with weighted BCov:
bcov.test(x = Dx, y = Dy, distance = TRUE, weight = TRUE)

################# Mutual Independence Test #################
x <- rnorm(30)
y <- (x > 0) * x + rnorm(30)
z <- (x <= 0) * x + rnorm(30)
data_list <- list(x, y, z)
bcov.test(data_list)

################# Mutual Independence Test for Meteorology data #################
data("meteorology")
bcov.test(meteorology)

################  Formula interface  ################
## independence test:
bcov.test(~ CONT + INTG, data = USJudgeRatings)
## mutual independence test:
bcov.test(~ CONT + INTG + DMNR, data = USJudgeRatings)
}
\references{
Jin, Zhu, Wenliang Pan, Wei Zheng, and Xueqin Wang (2018). Ball: An R package for detecting distribution difference and association in metric spaces. arXiv preprint arXiv:1811.03750. URL http://arxiv.org/abs/1811.03750.
}
\seealso{
\code{\link{bcov}}, \code{\link{bcor}}
}
\author{
Wenliang Pan, Xueqin Wang, Heping Zhang, Hongtu Zhu, Jin Zhu
}
