% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/blpFunctions.R
\name{estimateBLP}
\alias{estimateBLP}
\title{Performs a BLP demand estimation.}
\usage{
estimateBLP(Xlin, Xexo, Xrandom, instruments, demographics, shares, cdid,
  productData, demographicData, starting.guesses.theta2, starting.guesses.delta,
  solver.method = "BFGS", solver.control = list(), blp.control = list(),
  integration.control = list(), postEstimation.control = list(),
  printLevel = 0)
}
\arguments{
\item{Xlin}{character vector specifying the set of linear variables (variable names
must be included in \code{productData})}

\item{Xexo}{character vector specifying the set of exogenous variables (subset of \code{Xlin})}

\item{Xrandom}{character vector specifying the set of random coefficients (variable names
must be included in \code{productData})}

\item{instruments}{character vector specifying the set of instrumental variables (variable names
must be included in \code{productData})}

\item{demographics}{optional: character vector specifying the set of demographic variables (must be included as list entries in \code{demographicData})}

\item{shares}{character vector specifying observed market shares (variable name
must be included in \code{productData})}

\item{cdid}{character vector specifying the market identifier (variable name
must be included in \code{productData})}

\item{productData}{dataframe with product characteristics}

\item{demographicData}{optional: list with demographic data for each market (see details)}

\item{starting.guesses.theta2}{matrix with starting values for the optimization routine
(NA entries indicate the exclusion from estimation, i.e. the coefficient is assumed to be zero)}

\item{starting.guesses.delta}{optional: numeric vector with starting guesses for the mean utility}

\item{solver.method}{specifies the solver method in \code{optim} or \code{ucminf}}

\item{solver.control}{list of additional arguments for the optimization routines:
\describe{
\item{\code{solver.reltol}}{tolerance for the optimization routine }
\item{\code{solver.maxit}}{maximum iterations for the optimization routine}
\item{\code{...}}{further arguments passed to \code{optim} or \code{ucminf}}}}

\item{blp.control}{list of additional argruments for the BLP algorithm:
\describe{
\item{\code{inner.tol}}{tolerance for the contraction mapping}
\item{\code{inner.maxit}}{maximum iterations for the contraction mapping}}}

\item{integration.control}{list of parameters for the BLP integration problem:
\describe{
\item{\code{method}}{integration method}
\item{\code{amountNodes}}{integration accuracy for Monte Carlo based integration}
\item{\code{accuracyQuad}}{integration accuracy for integration by quadrature rules}
\item{\code{seed}}{seed for the draws of Monte Carlo based integration}
\item{\code{nodes}}{set of manually provided integration nodes}
\item{\code{weights}}{set of manually provided integration weights}
\item{\code{output}}{if \code{TRUE}, integration nodes and individual shares (sij) are available as output} }}

\item{postEstimation.control}{list of parameters specifying post estimation results:
\describe{
\item{\code{standardError}}{chose \code{robust} (default) or \code{nonRobust} }
\item{\code{extremumCheck}}{if \code{TRUE} (default), second derivatives are checked for the existence of minimum at the point estimate}
\item{\code{elasticities}}{character vector specifying the set of variables elasticities are calculated for} }}

\item{printLevel}{level of output information ranges from 1 (no GMM results) to 4 (every norm in the contraction mapping)}
}
\value{
Returns an object of class 'blp'. This object contains, among others,
all estimates for preference parameters and standard errors. Necessary information
for further post estimation analysis can be included as well.
}
\description{
Performs a BLP demand estimation.
}
\details{
The optimization routines are included in the packages \code{optim} and  \code{ucminf}. Only gradient based methods are supported.
The \code{ucminf} clones \code{MATLAB}s' standard trust region optimization routine, which turns out to be effective in
avoiding overflow problems in the BLP model. Valid arguments are \code{BFGS} , \code{BFGS_matlab}, \code{L-BFGS-B} or \code{CG}.

For solver options, the use of \code{solver.maxit} and \code{solver.rel} is recommended.
For conflicts of  \code{solver.maxit} and  \code{solver.reltol} and arguments of the respective solvers,
priority is given to the latter. Make sure that additionally provided solver control arguments are valid.

For demographics variables, list entries of \code{demographicData} must be named according to \code{demographics}.
Each list entry contains a dataframe with the draws for different markets and a variable
(according to the \code{cdid} argument) that allows to match the draws with the markets.

The logit model is used for elasticity calculation, if the variable of interest is not included in \code{Xrandom}.
Columns of the elasticity matrix contain the variables that are changed by 1\%, and rows contain effects on other products in the choice set.
}
\examples{
# Parameters
i<-1
K<-2
Xlin_example <-  c("price", "x1", "x2", "x3", "x4", "x5")
Xexo_example <- c("x1", "x2", "x3", "x4", "x5")
Xrandom_example <- paste0("x",1:K)
instruments_example <- paste0("iv",1:10)

# Data generation
BLP_data <- get.BLP.dataset(nmkt = 25, nbrn = 20,
                            Xlin = Xlin_example,
                            Xexo = Xexo_example,
                            Xrandom = Xrandom_example,
                            instruments = instruments_example,
                            true.parameters = list(Xlin.true.except.price = rep(0.2,5),
                                                   Xlin.true.price = -0.2, Xrandom.true = rep(2,K),
                                                   instrument.effects = rep(2,10),
                                                   instrument.Xexo.effects = rep(1,5)),
                            price.endogeneity = list( mean.xi = -2,
                                                      mean.eita = 0,
                                                      cov = cbind( c(1,0.7), c(0.7,1))),
                            printlevel = 0, seed = 5326 )

# Estimation
BLP_est<- estimateBLP(Xlin = Xlin_example,
                      Xrandom = Xrandom_example,
                      Xexo =  Xexo_example,
                      instruments = instruments_example,
                      shares = "shares",
                      cdid = "cdid",
                      productData = BLP_data,
                      starting.guesses.theta2 = rep(1,K),
                      solver.control = list(maxeval = 5000),
                      solver.method = "BFGS_matlab",

                      starting.guesses.delta =  rep(1, length(BLP_data$cdid)),
                      blp.control = list(inner.tol = 1e-6,
                                         inner.maxit = 5000),
                      integration.control= list(  method="MLHS",
                                                  amountNodes= 100,
                                                  seed= 3   ),
                      postEstimation.control= list(standardError = "robust",
                                                   extremumCheck = TRUE,
                                                   elasticities = "price"),
                      printLevel = 2)

# Show results
summary(BLP_est)
}
