# Copyright 2025 National Center of Neurology and Psychiatry
# BSD 3-Clause License (see LICENSE file)
#' @title Build SVM Classifier for Axon Image Feature Classification
#' @description
#' Trains a Support Vector Machine (SVM) classifier to distinguish between degenerative
#' and intact axon states using image feature data. Supports both direct path input
#' and interactive GUI selection.
#'
#' @param degenerate_path Character vector. Path to feature data for the "Degenerate" group
#'        (folder containing .txt files or direct file path). If NULL, GUI selection
#'        dialog will be shown. (default: NULL)
#' @param intact_path Character vector. Path to feature data for the "Intact" group
#'        (folder containing .txt files or direct file path). If NULL, GUI selection
#'        dialog will be shown. (default: NULL)
#' @param output_data_path Character. Path to save combined feature data (TSV).
#'        If NULL, GUI save dialog will be shown. (default: NULL)
#'        If a directory is specified, the file will be saved there with an autogenerated name.
#'        If a file name already exists, a unique name will be generated.(default: NULL)
#' @param output_model_path Character. Path to save trained SVM model (Rdata format).
#'        If NULL, GUI save dialog will be shown.
#'        If a directory is specified, the file will be saved there with an autogenerated name.
#'        If a file name already exists, a unique name will be generated.(default: NULL)
#' @param nCst Numeric. SVM cost parameter (C-value) controlling margin hardness.
#'        Higher values increase model complexity. (default: 3)
#' @param nGmm Numeric. SVM gamma parameter ( (\eqn{\gamma})-value) controlling RBF kernel width.
#'        Smaller values mean larger kernel radius. (default: 0.1)
#' @param nCrss Integer. Number of folds for cross-validation.
#'        Recommended values 5-10. (default: 5)
#'
#' @return Invisibly returns the trained SVM model object and produces the following outputs:
#' \itemize{
#'   \item Combined feature data file for machine learning (TSV).
#'   \item Trained SVM model file (Rdata format)
#' }
#'
#' @details
#' The function implements a complete machine learning pipeline:
#' \enumerate{
#'   \item \strong{Input handling}: If input paths are not specified, the user is prompted to select folders via a GUI dialog (cross-platform support for RStudio, tcltk, svDialogs, or manual input).
#'   \item \strong{Data preprocessing}: The function reads all .txt files in the specified folders, merges data, and labels them as "Degenerate" or "Intact".
#'   \item \strong{Feature selection}: Focuses on 7 key morphological features:
#'         \itemize{
#'           \item m.eccentricity (elliptical eccentricity)
#'           \item s.radius.sd (radial distribution uniformity)
#'           \item h.sva.s2 (Sum Variance: scale=2)
#'           \item h.idm.s1 (Inverse Difference Moment: local homogeneity scale=1)
#'           \item h.sen.s1 (Sum Entropy: structural complexity scale=1)
#'           \item m.majoraxis (major axis length)
#'         }
#'   \item \strong{Model training and evaluation}: Utilizes the e1071 package with a radial basis function (RBF) kernel. Cross-validation is performed.
#'   \item \strong{Result saving}: Exports the merged feature data and trained SVM model to disk, ensuring unique filenames if necessary.
#' }
#'
#'#' @section File Naming and Overwrite Policy:
#' \itemize{
#'   \item If the output path is an existing file, a unique name with a numeric suffix is generated to avoid overwriting.
#'   \item If the output path is an existing directory, the output file will be saved there with a default name (date-based).
#'   \item If the directory does not exist, the file will be saved in the current working directory.
#' }
#' All steps are performed automatically, requiring minimal user intervention.
#'
#' @section GUI Support:
#' \itemize{
#'   \item RStudio (rstudioapi), tcltk and svDialogs are supported for file/folder selection.
#'   \item If no GUI is available, the user is prompted to enter the path manually.
#' }
#'
#' @note
#' \itemize{
#'   \item Compatible with feature data from \code{\link{axDistmap}}
#'   \item Output files include timestamp in ISO 8601 format (YYYY-MM-DD)
#'   \item Model files can be reloaded using base::load()
#' }
#'
#' @examples
#' # Interactive mode with GUI prompts
#' # NOTE: This example requires a GUI environment for interactive folder selection.
#' if (interactive()){
#' axSvm()
#' }
#'
#'
#' # Direct path specification
#' \donttest{
#' deg_dir <- system.file("extdata", "Degenerate_txt", package = "AiES")
#' int_dir <- system.file("extdata", "Intact_txt", package = "AiES")
#' axSvm(degenerate_path = deg_dir, intact_path = int_dir,
#'       output_data_path = file.path(tempdir(), "svm_test_data.txt"),
#'       output_model_path = file.path(tempdir(), "svm_test_model.svm"))
#'
#' # Custom hyperparameters
#' deg_dir <- system.file("extdata", "Degenerate_txt", package = "AiES")
#' int_dir <- system.file("extdata", "Intact_txt", package = "AiES")
#' axSvm(degenerate_path = deg_dir, intact_path = int_dir,
#'      nCst = 5, nGmm = 0.05, nCrss = 10,
#'   output_data_path = tempdir(),       # specify directory only
#'   output_model_path = tempdir()       # specify directory only
#' )
#' }
#'
#' # Specify only output directory; default file names will be used
#' deg_dir <- system.file("extdata", "Degenerate_txt", package = "AiES")
#' int_dir <- system.file("extdata", "Intact_txt", package = "AiES")
#' axSvm(degenerate_path = deg_dir, intact_path = int_dir,
#'   output_data_path = tempdir(),       # specify directory only
#'   output_model_path = tempdir()       # specify directory only
#' )
#'
#' # In this case, output files will be saved as:
#' #   results/YYYY-MM-DD_Extracted_data_for_ML.txt
#' #   results/YYYY-MM-DD_AxClassifier.svm
#' # (YYYY-MM-DD is the current date)
#'
#' @importFrom data.table fread rbindlist fwrite
#' @importFrom e1071 svm
#'
#' @export
axSvm <- function(nCst = 3, nGmm = 0.1, nCrss=5,
                  degenerate_path = NULL,
                  intact_path = NULL,
                  output_data_path = tempdir(),
                  output_model_path = tempdir()
){


    ##############Definition of variable######################

    sdate <- Sys.Date()
    ftrTmp <- c("Group","m.eccentricity","s.radius.sd","h.sva.s2","h.idm.s1","h.sen.s1","m.majoraxis")
    txtGroup <- c("Degenerate","Intact")

    ###############Definition of function######################
    ##Function 1 Folder selection
    select_folder <- function(caption = "Select folder") {
        if (rstudioapi::isAvailable()) {
            return(rstudioapi::selectDirectory(caption = caption))
        } else if (requireNamespace("tcltk", quietly = TRUE)) {
            return(tcltk::tk_choose.dir(caption = caption))
        } else if (requireNamespace("svDialogs", quietly = TRUE)) {
            return(svDialogs::dlg_dir(title = caption)$res)
        } else {
            message("No GUI folder selection method available")
            return(NULL)
        }
    }

    ##Function 2 File selection
    select_files <- function(caption = "Select files") {
        if (.Platform$OS.type == "windows") {
            return(utils::choose.files(caption = caption, multi = TRUE))
        } else if (requireNamespace("tcltk", quietly = TRUE)) {
            return(tcltk::tk_choose.files(caption = caption, multi = TRUE))
        } else if (requireNamespace("svDialogs", quietly = TRUE)) {
            return(svDialogs::dlg_open(title = caption, multiple = TRUE)$res)
        } else {
            stop("No GUI file selection method available on this OS.")
        }
    }

    ##Function 3 Function to avoid duplicate file names
    generate_unique_filename <- function(filepath) {
        if (!file.exists(filepath)) return(filepath)
        base <- sub("(\\.[^.]*)$", "", filepath)   # before extension
        ext <- sub("^.*(\\.[^.]*)$", "\\1", filepath) # extension
        i <- 1
        new_filepath <- paste0(base, "_", i, ext)
        while (file.exists(new_filepath)) {
            i <- i + 1
            new_filepath <- paste0(base, "_", i, ext)
        }
        return(new_filepath)
    }

    ##Function 4 Function to specify output path
    output_path <- function(path, default_name, select_save_file_fun, caption = "Save file as") {
        # GUI or manual input
        if (is.null(path)) {
            resolved <- select_save_file_fun(default_name, caption = caption)
            if (is.null(resolved) || resolved == "") return(NULL)
            path <- resolved
        }
        # If an existing directory is selected, save it as default_name
        if (dir.exists(path)) {
            path <- file.path(path, default_name)
            path <- generate_unique_filename(path)
        } else if (file.exists(path)) {
            # Remove duplicates (if necessary)
            path <- generate_unique_filename(path)
        } else {
            # non-existent path
            folder_part <- dirname(path)
            if (!dir.exists(folder_part)) {
                warning(sprintf("Directory '%s' does not exist. Saving to current directory.", folder_part))
                path <- generate_unique_filename(default_name)
            }
            # If the folder exists, leave it as is (duplicate checked)
        }
        return(path)
    }

    # 1. Obtain input data
    if (is.null(degenerate_path)) {
        message("Select folder or .txt files for Degenerate condition")
        degenerate_path <- select_folder("Select Degenerate folder")
        if (is.na(degenerate_path) || is.null(degenerate_path)) return(message("No folder selected."))
        degenerate_files <- list.files(degenerate_path, pattern = "\\.txt$", full.names = TRUE)
    } else if (dir.exists(degenerate_path)) {
        degenerate_files <- list.files(degenerate_path, pattern = "\\.txt$", full.names = TRUE)
    } else {
        degenerate_files <- degenerate_path
    }
    if (length(degenerate_files) == 0) return(message("No degenerate files found."))

    if (is.null(intact_path)) {
        message("Select folder or .txt files for Intact condition")
        intact_path <- select_folder("Select Intact folder")
        if (is.na(intact_path) || is.null(intact_path)) return(message("No folder selected."))
        intact_files <- list.files(intact_path, pattern = "\\.txt$", full.names = TRUE)
    } else if (dir.exists(intact_path)) {
        intact_files <- list.files(intact_path, pattern = "\\.txt$", full.names = TRUE)
    } else {
        intact_files <- intact_path
    }
    if (length(intact_files) == 0) return(message("No intact files found."))


    # 2. Data reading (large-scale support)
    read_data <- function(files, group) {
        datalist <- lapply(files, function(f) {
            dt <- tryCatch(data.table::fread(f), error = function(e) NULL)
            if (is.null(dt)) return(NULL)
            dt$Group <- group
            dt
        })
        datalist <- datalist[!sapply(datalist, is.null)]
        if (length(datalist) == 0) return(NULL)
        data.table::rbindlist(datalist, fill = TRUE)
    }
    deg_data <- read_data(degenerate_files, "Degenerate")
    intact_data <- read_data(intact_files, "Intact")
    if (is.null(deg_data) || is.null(intact_data)) return(message("Failed to read data."))



    # 3. Data merging and feature selection
    dataSvm <- rbind(deg_data, intact_data)
    dataSvm <- dataSvm[, ..ftrTmp]
    dataSvm$Group <- as.factor(dataSvm$Group)
    rm(deg_data, intact_data); gc()

    # 4. SVM model construction
    svmModel <- e1071::svm(
        Group ~ .,
        data = dataSvm,
        method = "C-classification",
        probability = TRUE,
        kernel = "radial",
        gamma = nGmm,
        cost = nCst,
        cross = nCrss
    )

    # 5. Specify output file
    select_save_file <- function(default_name = "output.txt", caption = "Save file as") {
        # RStudio
        if (requireNamespace("rstudioapi", quietly = TRUE) && rstudioapi::isAvailable()) {
            path <- rstudioapi::selectFile(caption = caption, path = default_name, label = "Save", existing = FALSE)
            if (!is.null(path) && path != "") return(path)
        }
        # tcltk
        if (requireNamespace("tcltk", quietly = TRUE)) {
            path <- tcltk::tkgetSaveFile(initialfile = default_name, title = caption)
            if (!is.null(path) && path != "") return(path)
        }
        # svDialogs
        if (requireNamespace("svDialogs", quietly = TRUE)) {
            path <- svDialogs::dlg_save(default = default_name, title = caption)$res
            if (!is.null(path) && path != "") return(path)
        }
        # manual input
        cat(sprintf("Please enter the full path to save the file [%s]: ", default_name))
        path <- readline()
        if (path == "") path <- default_name
        return(path)
    }



    # Output data file save location
    default_data_name <- paste0(Sys.Date(), "_Extracted_data_for_ML.txt")
    output_data_path <- output_path(
        output_data_path,
        default_data_name,
        select_save_file,
        caption = "Save extracted data file"
    )
    if (is.null(output_data_path) || output_data_path == "") return(message("Canceled"))
    data.table::fwrite(dataSvm, file = output_data_path, sep = "\t")
    message("Extracted data saved: ", output_data_path)

    # Model file save location
    default_model_name <- paste0(Sys.Date(), "_AxClassifier.svm")
    output_model_path <- output_path(
        output_model_path,
        default_model_name,
        select_save_file,
        caption = "Save SVM model"
    )
    if (is.null(output_model_path) || output_model_path == "") return(message("Canceled"))
    save(svmModel, file = output_model_path)
    message("SVM model saved: ", output_model_path)


    invisible(svmModel)

}###end of function


