use crate::*;
use std::os::raw;




pub trait Rinternals: Types + Conversions {

    fn is_null(&self) -> bool {
        unsafe { Rf_isNull(self.get()) != 0 }
    }


    fn is_symbol(&self) -> bool {
        unsafe { Rf_isSymbol(self.get()) != 0 }
    }


    fn is_logical(&self) -> bool {
        unsafe { Rf_isLogical(self.get()) != 0 }
    }


    fn is_real(&self) -> bool {
        unsafe { Rf_isReal(self.get()) != 0 }
    }


    fn is_complex(&self) -> bool {
        unsafe { Rf_isComplex(self.get()) != 0 }
    }


    fn is_expressions(&self) -> bool {
        unsafe { Rf_isExpression(self.get()) != 0 }
    }


    fn is_environment(&self) -> bool {
        unsafe { Rf_isEnvironment(self.get()) != 0 }
    }


    fn is_promise(&self) -> bool {
        self.sexptype() == PROMSXP
    }


    fn is_string(&self) -> bool {
        unsafe { Rf_isString(self.get()) != 0 }
    }


    fn is_object(&self) -> bool {
        unsafe { Rf_isObject(self.get()) != 0 }
    }


    fn is_s4(&self) -> bool {
        unsafe { Rf_isS4(self.get()) != 0 }
    }


    fn is_external_pointer(&self) -> bool {
        self.rtype() == Rtype::ExternalPtr
    }


    fn get_current_srcref(val: i32) -> Robj {
        unsafe { Robj::from_sexp(R_GetCurrentSrcref(val as raw::c_int)) }
    }


    fn get_src_filename(&self) -> Robj {
        unsafe { Robj::from_sexp(R_GetSrcFilename(self.get())) }
    }


    fn as_character_vector(&self) -> Robj {
        unsafe { Robj::from_sexp(Rf_asChar(self.get())) }
    }


    fn coerce_vector(&self, sexptype: u32) -> Robj {
        single_threaded(|| unsafe {
            Robj::from_sexp(Rf_coerceVector(self.get(), sexptype as SEXPTYPE))
        })
    }


    fn pair_to_vector_list(&self) -> Robj {
        single_threaded(|| unsafe { Robj::from_sexp(Rf_PairToVectorList(self.get())) })
    }


    fn vector_to_pair_list(&self) -> Robj {
        single_threaded(|| unsafe { Robj::from_sexp(Rf_VectorToPairList(self.get())) })
    }


    fn as_character_factor(&self) -> Robj {
        single_threaded(|| unsafe { Robj::from_sexp(Rf_asCharacterFactor(self.get())) })
    }


    fn alloc_matrix(sexptype: SEXPTYPE, rows: i32, cols: i32) -> Robj {
        single_threaded(|| unsafe { Robj::from_sexp(Rf_allocMatrix(sexptype, rows, cols)) })
    }



    fn duplicate(&self) -> Robj {
        single_threaded(|| unsafe { Robj::from_sexp(Rf_duplicate(self.get())) })
    }
















    fn find_function<K: TryInto<Symbol, Error = Error>>(&self, key: K) -> Result<Robj> {
        let key: Symbol = key.try_into()?;
        if !self.is_environment() {
            return Err(Error::NotFound(key.into()));
        }
















        unsafe {
            let sexp = self.get();
            if let Ok(var) = catch_r_error(|| Rf_findFun(key.get(), sexp)) {
                Ok(Robj::from_sexp(var))
            } else {
                Err(Error::NotFound(key.into()))
            }
        }
    }



















    fn find_var<K: TryInto<Symbol, Error = Error>>(&self, key: K) -> Result<Robj> {
        let key: Symbol = key.try_into()?;
        if !self.is_environment() {
            return Err(Error::NotFound(key.into()));
        }
















        unsafe {
            let sexp = self.get();
            if let Ok(var) = catch_r_error(|| Rf_findVar(key.get(), sexp)) {
                if var != R_UnboundValue {
                    Ok(Robj::from_sexp(var))
                } else {
                    Err(Error::NotFound(key.into()))
                }
            } else {
                Err(Error::NotFound(key.into()))
            }
        }
    }










    fn eval_promise(&self) -> Result<Robj> {
        if self.is_promise() {
            self.as_promise().unwrap().eval()
        } else {
            Ok(self.as_robj().clone())
        }
    }


    fn ncols(&self) -> usize {
        unsafe { Rf_ncols(self.get()) as usize }
    }


    fn nrows(&self) -> usize {
        unsafe { Rf_nrows(self.get()) as usize }
    }


    #[doc(hidden)]
    unsafe fn make_external_ptr<T>(p: *mut T, prot: Robj) -> Robj {
        let type_name: Robj = std::any::type_name::<T>().into();
        Robj::from_sexp(single_threaded(|| {
            R_MakeExternalPtr(
                p as *mut ::std::os::raw::c_void,
                type_name.get(),
                prot.get(),
            )
        }))
    }


    #[doc(hidden)]
    unsafe fn external_ptr_addr<T>(&self) -> *mut T {
        R_ExternalPtrAddr(self.get()) as *mut T
    }


    #[doc(hidden)]
    unsafe fn external_ptr_tag(&self) -> Robj {
        Robj::from_sexp(R_ExternalPtrTag(self.get()))
    }


    #[doc(hidden)]
    unsafe fn external_ptr_protected(&self) -> Robj {
        Robj::from_sexp(R_ExternalPtrProtected(self.get()))
    }

    #[doc(hidden)]
    unsafe fn register_c_finalizer(&self, func: R_CFinalizer_t) {


        single_threaded(|| R_RegisterCFinalizerEx(self.get(), func, 1));
    }



    fn xlengthgets(&self, new_len: usize) -> Result<Robj> {
        unsafe {
            if self.is_vector() {
                Ok(single_threaded(|| {
                    Robj::from_sexp(Rf_xlengthgets(self.get(), new_len as R_xlen_t))
                }))
            } else {
                Err(Error::ExpectedVector(self.as_robj().clone()))
            }
        }
    }


    fn alloc_vector(sexptype: u32, len: usize) -> Robj {
        single_threaded(|| unsafe { Robj::from_sexp(Rf_allocVector(sexptype, len as R_xlen_t)) })
    }


    fn conformable(a: &Robj, b: &Robj) -> bool {
        single_threaded(|| unsafe { Rf_conformable(a.get(), b.get()) != 0 })
    }


    fn is_array(&self) -> bool {
        unsafe { Rf_isArray(self.get()) != 0 }
    }


    fn is_factor(&self) -> bool {
        unsafe { Rf_isFactor(self.get()) != 0 }
    }


    fn is_frame(&self) -> bool {
        unsafe { Rf_isFrame(self.get()) != 0 }
    }


    fn is_function(&self) -> bool {
        unsafe { Rf_isFunction(self.get()) != 0 }
    }


    fn is_integer(&self) -> bool {
        unsafe { Rf_isInteger(self.get()) != 0 }
    }


    fn is_language(&self) -> bool {
        unsafe { Rf_isLanguage(self.get()) != 0 }
    }


    fn is_pairlist(&self) -> bool {
        unsafe { Rf_isList(self.get()) != 0 }
    }


    fn is_matrix(&self) -> bool {
        unsafe { Rf_isMatrix(self.get()) != 0 }
    }


    fn is_list(&self) -> bool {
        unsafe { Rf_isNewList(self.get()) != 0 }
    }


    fn is_number(&self) -> bool {
        unsafe { Rf_isNumber(self.get()) != 0 }
    }


    fn is_primitive(&self) -> bool {
        unsafe { Rf_isPrimitive(self.get()) != 0 }
    }


    fn is_ts(&self) -> bool {
        unsafe { Rf_isTs(self.get()) != 0 }
    }


    fn is_user_binop(&self) -> bool {
        unsafe { Rf_isUserBinop(self.get()) != 0 }
    }


    fn is_valid_string(&self) -> bool {
        unsafe { Rf_isValidString(self.get()) != 0 }
    }


    fn is_valid_string_f(&self) -> bool {
        unsafe { Rf_isValidStringF(self.get()) != 0 }
    }


    fn is_vector(&self) -> bool {
        unsafe { Rf_isVector(self.get()) != 0 }
    }


    fn is_vector_atomic(&self) -> bool {
        unsafe { Rf_isVectorAtomic(self.get()) != 0 }
    }


    fn is_vector_list(&self) -> bool {
        unsafe { Rf_isVectorList(self.get()) != 0 }
    }


    fn is_vectorizable(&self) -> bool {
        unsafe { Rf_isVectorizable(self.get()) != 0 }
    }


    fn is_raw(&self) -> bool {
        self.rtype() == Rtype::Raw
    }


    fn is_char(&self) -> bool {
        self.rtype() == Rtype::Rstr
    }



    #[doc(hidden)]
    fn check_external_ptr_type<T>(&self) -> bool {
        if self.sexptype() == libR_sys::EXTPTRSXP {
            let tag = unsafe { self.external_ptr_tag() };
            if tag.as_str() == Some(std::any::type_name::<T>()) {
                return true;
            }
        }
        false
    }

    fn is_missing_arg(&self) -> bool {
        unsafe { self.get() == R_MissingArg }
    }

    fn is_unbound_value(&self) -> bool {
        unsafe { self.get() == R_UnboundValue }
    }

    fn is_package_env(&self) -> bool {
        unsafe { R_IsPackageEnv(self.get()) != 0 }
    }

    fn package_env_name(&self) -> Robj {
        unsafe { Robj::from_sexp(R_PackageEnvName(self.get())) }
    }

    fn is_namespace_env(&self) -> bool {
        unsafe { R_IsNamespaceEnv(self.get()) != 0 }
    }

    fn namespace_env_spec(&self) -> Robj {
        unsafe { Robj::from_sexp(R_NamespaceEnvSpec(self.get())) }
    }


    fn is_altrep(&self) -> bool {
        unsafe { ALTREP(self.get()) != 0 }
    }


    fn is_altinteger(&self) -> bool {
        unsafe { ALTREP(self.get()) != 0 && TYPEOF(self.get()) == INTSXP as i32 }
    }


    fn is_altreal(&self) -> bool {
        unsafe { ALTREP(self.get()) != 0 && TYPEOF(self.get()) == REALSXP as i32 }
    }


    fn is_altlogical(&self) -> bool {
        unsafe { ALTREP(self.get()) != 0 && TYPEOF(self.get()) == LGLSXP as i32 }
    }


    fn is_altraw(&self) -> bool {
        unsafe { ALTREP(self.get()) != 0 && TYPEOF(self.get()) == RAWSXP as i32 }
    }


    fn is_altstring(&self) -> bool {
        unsafe { ALTREP(self.get()) != 0 && TYPEOF(self.get()) == STRSXP as i32 }
    }


    #[cfg(use_r_altlist)]
    fn is_altlist(&self) -> bool {
        unsafe { ALTREP(self.get()) != 0 && TYPEOF(self.get()) == VECSXP as i32 }
    }


    fn deparse(&self) -> Result<String> {
        use crate as extendr_api;
        let strings: Strings = call!("deparse", self.as_robj())?.try_into()?;
        if strings.len() == 1 {
            Ok(String::from(strings.elt(0).as_str()))
        } else {
            Ok(strings
                .iter()
                .map(|s| s.as_str())
                .collect::<Vec<_>>()
                .join(""))
        }
    }
}

impl Rinternals for Robj {}
