% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/windowed_scale_index.R
\name{windowed_scale_index}
\alias{windowed_scale_index}
\title{Windowed scale index}
\usage{
windowed_scale_index(signal,
                            dt = 1,
                            scales = NULL,
                            powerscales = TRUE,
                            s1 = NULL,
                            windowrad = NULL,
                            delta_t = NULL,
                            wname = c("MORLET", "DOG", "PAUL", "HAAR", "HAAR2"),
                            wparam = NULL,
                            waverad = NULL,
                            border_effects = c("BE", "INNER", "PER", "SYM"),
                            makefigure = TRUE,
                            time_values = NULL,
                            figureperiod = TRUE)
}
\arguments{
\item{signal}{A vector containing the signal whose windowed scale indices are wanted.}

\item{dt}{Numeric. The time step of the signal.}

\item{scales}{A vector containing the wavelet scales at wich the windowed scalograms
are computed. This can be either a vector with all the scales, or (if
\code{powerscales} is TRUE) following Torrence and Compo 1998, a vector of three
elements with the minimum scale, the maximum scale and the number of suboctaves per
octave. If NULL, they are automatically computed.}

\item{powerscales}{Logical. If TRUE (default), construct power 2 scales from
\code{scales}. If \code{scales} is NULL, they are automatically computed.}

\item{s1}{A vector containing the scales \eqn{s_1}. The windowed scale indices are
computed in the intervals \eqn{[s_0,s_1]}, where \eqn{s_0} is the minimum scale in
\code{scales}.}

\item{windowrad}{Integer. Time radius for the windows, measured in \code{dt}. By
default, it is set to \eqn{celing(length(signal1) / 20)}.}

\item{delta_t}{Integer. Increment of time for the construction of windows central
times, measured in \code{dt}. By default, it is set to
\eqn{celing(length(signal) / 256)}.}

\item{wname}{A string, equal to "MORLET", "DOG", "PAUL", "HAAR" or "HAAR2". The
difference between "HAAR" and "HAAR2" is that "HAAR2" is more accurate but slower.}

\item{wparam}{The corresponding nondimensional parameter for the wavelet function
(Morlet, DoG or Paul).}

\item{waverad}{Numeric. The radius of the wavelet used in the computations for the cone
of influence. If it is not specified, it is computed by \code{wavelet_radius} for DoG
and Paul wavelets. For Haar and Morlet it is assumed to be 1 and 3 respectively.}

\item{border_effects}{A string, equal to "BE", "INNER", "PER" or "SYM", which indicates
how to manage the border effects which arise usually when a convolution is performed on
finite-lenght signals.
\itemize{
\item "BE": With border effects, padding time series with zeroes.
\item "INNER": Normalized inner scalogram with security margin adapted for each
different scale.
\item "PER": With border effects, using boundary wavelets (periodization of the
original time series).
\item "SYM": With border effects, using a symmetric catenation of the original time
series.
}}

\item{makefigure}{Logical. If TRUE (default), a figure with the windowed scale indices
is plotted.}

\item{time_values}{A numerical vector of length \code{length(signal)} containing custom
time values for the figure. If NULL (default), it will be computed starting at 0.}

\item{figureperiod}{Logical. If TRUE (default), periods are used in the figure instead
of scales.}
}
\value{
A list with the following fields:
\itemize{
\item \code{wsi}: A matrix of size \code{length(tcentral)} x \code{length(s1)}
containing the values of the corresponding windowed scale indices.
\item \code{s1}: The vector of scales \eqn{s_1}.
\item \code{smax}: A matrix of size \code{length(tcentral)} x \code{length(s1)}
containing the scales \eqn{s_{max}}.
\item \code{smin}: A matrix of size \code{length(tcentral)} x \code{length(s1)}
containing the scales \eqn{s_{min}}.
\item \code{scalog_smax}: A matrix of size \code{length(tcentral)} x \code{length(s1)}
containing the values of the corresponding scalograms at scales \eqn{s_{max}}.
\item \code{scalog_smin}: A matrix of size \code{length(tcentral)} x \code{length(s1)}
containing the values of the corresponding scalograms at scales \eqn{s_{min}}.
\item \code{tcentral}: The vector of central times used in the computation of
\code{wsi}.
\item \code{fourier_factor}: A factor for converting scales to periods.
\item \code{coi_maxscale}: A vector of length \code{length(tcentral)} containing the
values of the maximum scale from which there are border effects.
}
}
\description{
This function computes the windowed scale indices of a signal in the scale
interval \eqn{[s_0,s_1]}, for a given set of scale parameters \eqn{s_1} and taking
\eqn{s_0} as the minimum scale (see Benítez et al. 2010).

The windowed scale index of a signal in the scale interval \eqn{[s_0,s_1]} centered at
time \eqn{tc} and with time windows radius \code{windowrad} is given by the quotient
\deqn{\frac{WS_{windowrad}(tc,s_{min})}{WS_{windowrad}(tc,s_{max})},}{WS_{windowrad}
(tc,s_{min})/WS_{windowrad}(tc,s_{max}),}
where \eqn{WS_{windowrad}} is the corresponding windowed scalogram with time windows
radius \code{windowrad}, \eqn{s_{max} in [s_0,s_1]} is the smallest scale such that
\eqn{WS_{windowrad}(tc,s)\le WS_{windowrad}(tc,s_{max})} for all \eqn{s in [s_0,s_1]},
and \eqn{s_{min} in [s_{max},2s_1]} is the smallest scale such that
\eqn{WS_{windowrad}(tc,s_{min})\le WS_{windowrad}(tc,s)} for all
\eqn{s in [s_{max},2s_1]}.
}
\section{References}{


R. Benítez, V. J. Bolós, M. E. Ramírez. A wavelet-based tool for studying
non-periodicity. Comput. Math. Appl. 60 (2010), no. 3, 634-641.
}

\examples{
dt <- 0.1
time <- seq(0, 50, dt)
signal <- c(sin(pi * time), sin(pi * time / 2))
# First, we try with default s1 scales (a vector with a wide range of values for s1).
wsi_full <- windowed_scale_index(signal = signal, dt = dt, figureperiod = FALSE)
# Next, we choose a meaningful s1 value, greater than all relevant scales.
wsi <- windowed_scale_index(signal = signal, dt = dt, s1 = 4, figureperiod = FALSE)

}
