% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MAINest.R
\name{fitGSMAR}
\alias{fitGSMAR}
\title{Estimate Gaussian or Student's t Mixture Autoregressive model}
\usage{
fitGSMAR(
  data,
  p,
  M,
  model = c("GMAR", "StMAR", "G-StMAR"),
  restricted = FALSE,
  constraints = NULL,
  conditional = TRUE,
  parametrization = c("intercept", "mean"),
  ncalls = round(10 + 9 * log(sum(M))),
  ncores = min(2, ncalls, parallel::detectCores()),
  maxit = 300,
  seeds = NULL,
  printRes = TRUE,
  runTests = FALSE,
  ...
)
}
\arguments{
\item{data}{a numeric vector or class \code{'ts'} object containing the data. \code{NA} values are not supported.}

\item{p}{a positive integer specifying the autoregressive order of the model.}

\item{M}{\describe{
  \item{For \strong{GMAR} and \strong{StMAR} models:}{a positive integer specifying the number of mixture components.}
  \item{For \strong{G-StMAR} models:}{a size (2x1) integer vector specifying the number of \emph{GMAR type} components \code{M1} in the
   first element and \emph{StMAR type} components \code{M2} in the second element. The total number of mixture components is \code{M=M1+M2}.}
}}

\item{model}{is "GMAR", "StMAR", or "G-StMAR" model considered? In the G-StMAR model, the first \code{M1} components
are \emph{GMAR type} and the rest \code{M2} components are \emph{StMAR type}.}

\item{restricted}{a logical argument stating whether the AR coefficients \eqn{\phi_{m,1},...,\phi_{m,p}} are restricted
to be the same for all regimes.}

\item{constraints}{specifies linear constraints applied to the autoregressive parameters.
\describe{
\item{For \strong{non-restricted} models:}{a list of size \eqn{(pxq_{m})} constraint matrices \strong{\eqn{C_{m}}} of full column rank
  satisfying \strong{\eqn{\phi_{m}}}\eqn{=}\strong{\eqn{C_{m}\psi_{m}}} for all \eqn{m=1,...,M}, where
  \strong{\eqn{\phi_{m}}}\eqn{=(\phi_{m,1},...,\phi_{m,p})} and \strong{\eqn{\psi_{m}}}\eqn{=(\psi_{m,1},...,\psi_{m,q_{m}})}.}
\item{For \strong{restricted} models:}{a size \eqn{(pxq)} constraint matrix \strong{\eqn{C}} of full column rank satisfying
  \strong{\eqn{\phi}}\eqn{=}\strong{\eqn{C\psi}}, where \strong{\eqn{\phi}}\eqn{=(\phi_{1},...,\phi_{p})} and
  \strong{\eqn{\psi}}\eqn{=\psi_{1},...,\psi_{q}}.}
}
Symbol \eqn{\phi} denotes an AR coefficient. Note that regardless of any constraints, the nominal autoregressive order
is always \code{p} for all regimes.
Ignore or set to \code{NULL} if applying linear constraints is \strong{not} desired.}

\item{conditional}{a logical argument specifying whether the conditional or exact log-likelihood function should be used.}

\item{parametrization}{is the model parametrized with the "intercepts" \eqn{\phi_{m,0}} or
"means" \eqn{\mu_m = \phi_{m,0}/(1-\sum\phi_{i,m})}?}

\item{ncalls}{a positive integer specifying how many rounds of estimation should be conducted.
The estimation results may vary from round to round because of multimodality of the log-likelihood function
and the randomness associated with the genetic algorithm.}

\item{ncores}{the number of CPU cores to be used in the estimation process.}

\item{maxit}{the maximum number of iterations for the variable metric algorithm.}

\item{seeds}{a length \code{ncalls} vector containing the random number generator seed for each call to the genetic algorithm,
or \code{NULL} for not initializing the seed. Exists for the purpose of creating reproducible results.}

\item{printRes}{should the estimation results be printed?}

\item{runTests}{should quantile residuals tests be performed after the estimation?}

\item{...}{additional settings passed to the function \code{GAfit} employing the genetic algorithm.}
}
\value{
Returns an object of class \code{'gsmar'} defining the estimated GMAR, StMAR or G-StMAR model. The returned object contains
  estimated mixing weights, some conditional and unconditional moments, quantile residuals, and quantile residual test results
  if the tests were performed. Note that the first p observations are taken as the initial values so the mixing weights, conditional
  moments, and quantile residuals start from the p+1:th observation (interpreted as t=1).In addition, the returned object contains
  the estimates and log-likelihood values from all of the estimation rounds.
  The estimated parameter vector can be obtained as \code{gsmar$params} (and the corresponding approximate standard errors as
  \code{gsmar$std_errors}) and it's...
 \describe{
   \item{For \strong{non-restricted} models:}{
     \describe{
       \item{For \strong{GMAR} model:}{Size \eqn{(M(p+3)-1x1)} vector \strong{\eqn{\theta}}\eqn{=}(\strong{\eqn{\upsilon_{1}}},...,\strong{\eqn{\upsilon_{M}}},
         \eqn{\alpha_{1},...,\alpha_{M-1}}), where \strong{\eqn{\upsilon_{m}}}\eqn{=(\phi_{m,0},}\strong{\eqn{\phi_{m}}}\eqn{,
         \sigma_{m}^2)} and \strong{\eqn{\phi_{m}}}=\eqn{(\phi_{m,1},...,\phi_{m,p}), m=1,...,M}.}
       \item{For \strong{StMAR} model:}{Size \eqn{(M(p+4)-1x1)} vector (\strong{\eqn{\theta, \nu}})\eqn{=}(\strong{\eqn{\upsilon_{1}}},...,\strong{\eqn{\upsilon_{M}}},
         \eqn{\alpha_{1},...,\alpha_{M-1}, \nu_{1},...,\nu_{M}}).}
       \item{For \strong{G-StMAR} model:}{Size \eqn{(M(p+3)+M2-1x1)} vector (\strong{\eqn{\theta, \nu}})\eqn{=}(\strong{\eqn{\upsilon_{1}}},...,\strong{\eqn{\upsilon_{M}}},
         \eqn{\alpha_{1},...,\alpha_{M-1}, \nu_{M1+1},...,\nu_{M}}).}
       \item{With \strong{linear constraints}:}{Replace the vectors \strong{\eqn{\phi_{m}}} with vectors \strong{\eqn{\psi_{m}}} and provide a  list of constraint
         matrices \strong{C} that satisfy \strong{\eqn{\phi_{m}}}\eqn{=}\strong{\eqn{C_{m}\psi_{m}}} for all \eqn{m=1,...,M}, where
         \strong{\eqn{\psi_{m}}}\eqn{=(\psi_{m,1},...,\psi_{m,q_{m}})}.}
     }
   }
   \item{For \strong{restricted} models:}{
     \describe{
       \item{For \strong{GMAR} model:}{Size \eqn{(3M+p-1x1)} vector \strong{\eqn{\theta}}\eqn{=(\phi_{1,0},...,\phi_{M,0},}\strong{\eqn{\phi}}\eqn{,
         \sigma_{1}^2,...,\sigma_{M}^2,\alpha_{1},...,\alpha_{M-1})}, where \strong{\eqn{\phi}}=\eqn{(\phi_{1},...,\phi_{M})}.}
       \item{For \strong{StMAR} model:}{Size \eqn{(4M+p-1x1)} vector (\strong{\eqn{\theta, \nu}})\eqn{=(\phi_{1,0},...,\phi_{M,0},}\strong{\eqn{\phi}}\eqn{,
         \sigma_{1}^2,...,\sigma_{M}^2,\alpha_{1},...,\alpha_{M-1}, \nu_{1},...,\nu_{M})}.}
       \item{For \strong{G-StMAR} model:}{Size \eqn{(3M+M2+p-1x1)} vector (\strong{\eqn{\theta, \nu}})\eqn{=(\phi_{1,0},...,\phi_{M,0},}\strong{\eqn{\phi}}\eqn{,
         \sigma_{1}^2,...,\sigma_{M}^2,\alpha_{1},...,\alpha_{M-1}, \nu_{M1+1},...,\nu_{M})}.}
       \item{With \strong{linear constraints}:}{Replace the vector \strong{\eqn{\phi}} with vector \strong{\eqn{\psi}} and provide a constraint matrix
         \strong{\eqn{C}} that satisfies \strong{\eqn{\phi}}\eqn{=}\strong{\eqn{C\psi}}, where
         \strong{\eqn{\psi}}\eqn{=(\psi_{1},...,\psi_{q})}.}
     }
   }
 }
 Symbol \eqn{\phi} denotes an AR coefficient, \eqn{\sigma^2} a variance, \eqn{\alpha} a mixing weight, and \eqn{\nu} a degrees of
 freedom parameter. If \code{parametrization=="mean"} just replace each intercept term \eqn{\phi_{m,0}} with regimewise mean
 \eqn{\mu_m = \phi_{m,0}/(1-\sum\phi_{i,m})}. In the \strong{G-StMAR} model, the first \code{M1} components are \emph{GMAR type}
 and the rest \code{M2} components are \emph{StMAR type}.
 Note that in the case \strong{M=1} the parameter \eqn{\alpha} is dropped, and in the case of \strong{StMAR} or \strong{G-StMAR} model
 the degrees of freedom parameters \eqn{\nu_{m}} have to be larger than \eqn{2}.
}
\description{
\code{fitGSMAR} estimates GMAR, StMAR, or G-StMAR model in two phases. In the first phase, a genetic algorithm is employed
  to find starting values for a gradient based method. In the second phase, the gradient based variable metric algorithm is utilized to
  accurately converge to a local maximum or a saddle point near each starting value. Parallel computing is used to conduct multiple
  rounds of estimations in parallel.
}
\details{
Because of complexity and multimodality of the log-likelihood function, it's \strong{not guaranteed} that the estimation
 algorithm will end up in the global maximum point. It's often expected that most of the estimation rounds will end up in
 some local maximum point instead, and therefore a number of estimation rounds is required for reliable results. Because
 of the nature of the models, the estimation may fail particularly in the cases where the number of mixture components is
 chosen too large. Note that the genetic algorithm is designed to avoid solutions with mixing weights of some regimes
 too close to zero at almost all times ('redudant regimes') but the settings can, however, be adjusted (see ?GAfit).

 If the iteration limit for the variable metric algorithm (\code{maxit}) is reached, one can continue the estimation by
 iterating more with the function \code{iterate_more}.

 The core of the genetic algorithm is mostly based on the description by \emph{Dorsey and Mayer (1995)}. It utilizes
 a slightly modified version the individually adaptive crossover and mutation rates described by \emph{Patnaik and Srinivas (1994)}
 and employs (50\%) fitness inheritance discussed by \emph{Smith, Dike and Stegmann (1995)}. Large (in absolute value) but stationary
 AR parameter values are generated with the algorithm proposed by Monahan (1984).

 The variable metric algorithm (or quasi-Newton method, Nash (1990, algorithm 21)) used in the second phase is implemented
 with function the \code{optim} from the package \code{stats}.

 Some mixture components of the StMAR model may sometimes get very large estimates for the degrees of freedom parameters. Such parameters
 are weakly identified and induce various numerical problems. However, mixture components with large degree of freedom parameters are
 similar to the mixture components of the GMAR model. It's hence advisable to further estimate a G-StMAR model by allowing the mixture
 components with large degrees of freedom parameter estimates to be GMAR type.
}
\section{S3 methods}{

 The following S3 methods are supported for class \code{'gsmar'} objects: \code{print}, \code{summary}, \code{plot},
 \code{logLik}, \code{residuals}.
}

\section{Suggested packages}{

 For faster evaluation of the quantile residuals for StMAR and G-StMAR models, install the suggested package "gsl".
 Note that for large StMAR and G-StMAR models with large data, performing the quantile residual tests may take
 significantly long time without the package "gsl".
}

\examples{
\donttest{
# These are long running examples that use parallel computing

# GMAR model
fit12 <- fitGSMAR(simudata, p=1, M=2, model="GMAR")
summary(fit12)
plot(fit12)

# StMAR model
fit42 <- fitGSMAR(data=T10Y1Y, p=4, M=2, model="StMAR")
fit42
summary(fit42)
plot(fit42)

# Restricted StMAR model: plot also the individual statistics with
# their approximate critical bounds using the given data
fit42r <- fitGSMAR(T10Y1Y, 4, 2, model="StMAR", restricted=TRUE)
fit42r
plot(fit42)

# Non-mixture version of StMAR model
fit101t <- fitGSMAR(T10Y1Y, 10, 1, model="StMAR", ncores=1, ncalls=1)
diagnosticPlot(fit101t)

# G-StMAR model with one GMAR type and one StMAR type regime
fit42g <- fitGSMAR(T10Y1Y, 4, M=c(1, 1), model="G-StMAR")
diagnosticPlot(fit42g)

# GMAR model; seeds for rerpoducibility
fit43gm <- fitGSMAR(T10Y1Y, 4, M=3, model="GMAR", ncalls=16,
  seeds=1:16)
fit43gm

# Restricted GMAR model
fit43gmr <- fitGSMAR(T10Y1Y, 4, M=3, model="GMAR", ncalls=12,
  restricted=TRUE, seeds=1:12)
fit43gmr


# The following three examples demonstrate how to apply linear constraints
# to the AR parameters.

# Two-regime GMAR p=2 model with the second AR coeffiecient of
# of the second regime contrained to zero.
constraints <- list(diag(1, ncol=2, nrow=2), as.matrix(c(1, 0)))
fit22c <- fitGSMAR(T10Y1Y, 2, 2, constraints=constraints)
fit22c

# Such constrained StMAR(3, 1) model that the second order AR coefficient
# is constrained to zero.
constraints <- list(matrix(c(1, 0, 0, 0, 0, 1), ncol=2))
fit31tc <- fitGSMAR(T10Y1Y, 3, 1, model="StMAR", constraints=constraints)
fit31tc

# Such StMAR(3,2) that the AR coefficients are restricted to be
# the same for both regimes and that the second AR coefficients are
# constrained to zero.
fit32rc <- fitGSMAR(T10Y1Y, 3, 2, model="StMAR", restricted=TRUE,
 constraints=matrix(c(1, 0, 0, 0, 0, 1), ncol=2))
fit32rc
}
}
\references{
\itemize{
   \item Dorsey R. E. and Mayer W. J. 1995. Genetic algorithms for estimation problems with multiple optima,
         nondifferentiability, and other irregular features. \emph{Journal of Business & Economic Statistics},
         \strong{13}, 53-66.
   \item Kalliovirta L., Meitz M. and Saikkonen P. 2015. Gaussian Mixture Autoregressive model for univariate time series.
         \emph{Journal of Time Series Analysis}, \strong{36}, 247-266.
   \item Meitz M., Preve D., Saikkonen P. 2018. A mixture autoregressive model based on Student's t-distribution.
         arXiv:1805.04010 \strong{[econ.EM]}.
   \item Monahan J.F. 1984. A Note on Enforcing Stationarity in Autoregressive-Moving Average Models.
         \emph{Biometrica} \strong{71}, 403-404.
   \item Nash J. 1990. Compact Numerical Methods for Computers. Linear algebra and Function Minimization.
         \emph{Adam Hilger}.
   \item Patnaik L.M. and Srinivas M. 1994. Adaptive Probabilities of Crossover and Mutation in Genetic Algorithms.
         \emph{Transactions on Systems, Man and Cybernetics} \strong{24}, 656-667.
   \item Smith R.E., Dike B.A., Stegmann S.A. 1995. Fitness inheritance in genetic algorithms.
         \emph{Proceedings of the 1995 ACM Symposium on Applied Computing}, 345-350.
   \item Virolainen S. 2020. A mixture autoregressive model based on Gaussian and Student's t-distribution.	arXiv:2003.05221 [econ.EM].
 }
}
\seealso{
\code{\link{GSMAR}}, \code{\link{iterate_more}}, , \code{\link{stmar_to_gstmar}}, \code{\link{add_data}},
 \code{\link{profile_logliks}}, \code{\link{swap_parametrization}}, \code{\link{get_gradient}}, \code{\link{simulateGSMAR}}, \code{\link{predict.gsmar}},
  \code{\link{diagnosticPlot}}, \code{\link{quantileResidualTests}}, \code{\link{condMoments}}, \code{\link{uncondMoments}}
}
