% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ipe.R
\name{ipe}
\alias{ipe}
\title{Iterative Parameter Estimation (IPE) for Treatment Switching}
\usage{
ipe(
  data,
  stratum = "",
  time = "time",
  event = "event",
  treat = "treat",
  rx = "rx",
  censor_time = "censor_time",
  base_cov = "",
  aft_dist = "weibull",
  strata_main_effect_only = 1,
  treat_modifier = 1,
  recensor = TRUE,
  admin_recensor_only = FALSE,
  autoswitch = TRUE,
  alpha = 0.05,
  ties = "efron",
  tol = 1e-06,
  boot = FALSE,
  n_boot = 1000,
  seed = NA
)
}
\arguments{
\item{data}{The input data frame that contains the following variables:
\itemize{
\item \code{stratum}: The stratum.
\item \code{time}: The survival time for right censored data.
\item \code{event}: The event indicator, 1=event, 0=no event.
\item \code{treat}: The randomized treatment indicator, 1=treatment,
0=control.
\item \code{rx}: The proportion of time on active treatment.
\item \code{censor_time}: The administrative censoring time. It should
be provided for all subjects including those who had events.
\item \code{base_cov}: The baseline covariates (excluding treat).
}}

\item{stratum}{The name(s) of the stratum variable(s) in the input data.}

\item{time}{The name of the time variable in the input data.}

\item{event}{The name of the event variable in the input data.}

\item{treat}{The name of the treatment variable in the input data.}

\item{rx}{The name of the rx variable in the input data.}

\item{censor_time}{The name of the censor_time variable in the input data.}

\item{base_cov}{The vector of names of baseline covariates (excluding
treat) in the input data.}

\item{aft_dist}{The assumed distribution for time to event for the AFT
model. Options include "exponential", "weibull", "loglogistic", and
"lognormal".}

\item{strata_main_effect_only}{Whether to only include the strata main
effects in the AFT model. Defaults to \code{TRUE}, otherwise all
possible strata combinations will be considered in the AFT model.}

\item{treat_modifier}{The optional sensitivity parameter for the
constant treatment effect assumption.}

\item{recensor}{Whether to apply recensoring to counter-factual
survival times. Defaults to \code{TRUE}.}

\item{admin_recensor_only}{Whether to apply recensoring to administrative
censoring time only. Defaults to \code{FALSE}, in which case,
recensoring will be applied to the actual censoring time for dropouts.}

\item{autoswitch}{Whether to exclude recensoring for treatment arms
with no switching. Defaults to \code{TRUE}.}

\item{alpha}{The significance level to calculate confidence intervals.}

\item{ties}{The method for handling ties in the Cox model, either
"breslow" or "efron" (default).}

\item{tol}{The desired accuracy (convergence tolerance) for \code{psi}.}

\item{boot}{Whether to use bootstrap to obtain the confidence
interval for hazard ratio. Defaults to \code{FALSE}, in which case,
the confidence interval will be constructed to match the log-rank
test p-value.}

\item{n_boot}{The number of bootstrap samples.}

\item{seed}{The seed to reproduce the simulation results.
The seed from the environment will be used if left unspecified.}
}
\value{
A list with the following components:
\itemize{
\item \code{psi}: The estimated causal parameter.
\item \code{psi_CI}: The confidence interval for \code{psi}.
\item \code{psi_CI_type}: The type of confidence interval for \code{psi},
i.e., "log-rank p-value" or "bootstrap".
\item \code{Sstar}: A data frame containing the counter-factual unswitched
survival times and the event indicators.
\item \code{kmstar}: A data frame containing the Kaplan-Meier estimates
based on the counter-factual unswitched survival times by treatment
arm.
\item \code{logrank_pvalue}: The two-sided p-value of the log-rank test
based on the treatment policy strategy.
\item \code{cox_pvalue}: The two-sided p-value for treatment effect based on
the Cox model.
\item \code{hr}: The estimated hazard ratio from the Cox model.
\item \code{hr_CI}: The confidence interval for hazard ratio.
\item \code{hr_CI_type}: The type of confidence interval for hazard ratio,
either "log-rank p-value" or "bootstrap".
\item \code{settings}: A list with the following components:
\itemize{
\item \code{aft_dist}: The distribution for time to event for the AFT
model.
\item \code{strata_main_effect_only}: Whether to only include the strata
main effects in the AFT model.
\item \code{treat_modifier}: The sensitivity parameter for the constant
treatment effect assumption.
\item \code{recensor}: Whether to apply recensoring to counter-factual
survival times.
\item \code{admin_recensor_only}: Whether to apply recensoring to
administrative censoring time only.
\item \code{autoswitch}: Whether to exclude recensoring for treatment
arms with no switching.
\item \code{alpha}: The significance level to calculate confidence
intervals.
\item \code{ties}: The method for handling ties in the Cox model.
\item \code{tol}: The desired accuracy (convergence tolerance).
\item \code{boot}: Whether to use bootstrap to obtain the confidence
interval for hazard ratio.
\item \code{n_boot}: The number of bootstrap samples.
\item \code{seed}: The seed to reproduce the bootstrap results.
}
\item \code{hr_boots}: The bootstrap hazard ratio estimates if \code{boot} is
\code{TRUE}.
\item \code{psi_boots}: The bootstrap \code{psi} estimates if \code{boot} is
\code{TRUE}.
}
}
\description{
Obtains the causal parameter estimate from the
accelerated failure-time (AFT) model and the hazard ratio estimate
from the Cox model to account for treatment switching.
}
\details{
We use the following steps to obtain the hazard ratio estimate
and confidence interval had there been no treatment switching:
\itemize{
\item Use IPE to estimate the causal parameter \eqn{\psi} based on the AFT
model for counter-factual survival times for both arms:
\eqn{U_{i,\psi} = T_{C_i} + e^{\psi}T_{E_i}} for the control arm, and
\eqn{V_{i,\psi} = T_{E_i} + e^{-\psi}T_{C_i}} for the experimental arm.
\item Fit the Cox proportional hazards model to the counter-factual survival
times to obtain the hazard ratio estimate.
\item Use either the log-rank test p-value for the treatment policy strategy
or bootstrap to construct the confidence interval for hazard ratio.
}
}
\examples{

library(dplyr)

# Example 1: one-way treatment switching (control to active)

data <- immdef \%>\% mutate(rx = 1-xoyrs/progyrs)

fit1 <- ipe(
  data, time = "progyrs", event = "prog", treat = "imm", 
  rx = "rx", censor_time = "censyrs", aft_dist = "weibull",
  boot = FALSE)

c(fit1$hr, fit1$hr_CI)

# Example 2: two-way treatment switching (illustration only)

# the eventual survival time
shilong1 <- shilong \%>\%
  arrange(bras.f, id, tstop) \%>\%
  group_by(bras.f, id) \%>\%
  filter(row_number() == n()) \%>\%
  select(-c("ps", "ttc", "tran"))

shilong2 <- shilong1 \%>\%
  mutate(rx = ifelse(co, ifelse(bras.f == "MTA", dco/ady, 
                                1 - dco/ady),
                     ifelse(bras.f == "MTA", 1, 0)))

fit2 <- ipe(
  shilong2, time = "tstop", event = "event",
  treat = "bras.f", rx = "rx", censor_time = "dcut",
  base_cov = c("agerand", "sex.f", "tt_Lnum", "rmh_alea.c",
               "pathway.f"),
  aft_dist = "weibull", boot = FALSE)

c(fit2$hr, fit2$hr_CI)

}
\references{
Michael Branson and John Whitehead.
Estimating a treatment effect in survival studies in which patients
switch treatment.
Statistics in Medicine. 2002;21:2449-2463.

Ian R White.
Letter to the Editor: Estimating treatment effects in randomized
trials with treatment switching.
Statistics in Medicine. 2006;25:1619-1622.
}
\author{
Kaifeng Lu, \email{kaifenglu@gmail.com}
}
