#' Print Trello API Tokens
#'
#' Objects of class `"Trello_API_token"` posses some additional attributes to
#' make it easier to see the app name, its scope and expiration. If you want to
#' print the bare token object as generated by `[httr::oauth1.0_token()]`, call
#' its own print method, eg: `my_token$print()`.
#'
#' @param x An object of class `"Trello_API_token"`.
#' @param ... Unused.
#'
#' @export

print.Trello_API_token = function(x, ...) {

  cat("<Trello API token>\n")

  cat(" app name:   ", x$app$appname, "\n")
  cat(" permissions:", get_scope(x), "\n")
  cat(" approved:   ", get_time_issued(x), "\n")
  cat(" expires:    ", get_time_expiry(x))

  return(invisible(x))
}

get_scope = function(Trello_API_token) {

  paste(attr(Trello_API_token, which = "scope", exact = TRUE),
        collapse = ",")

}

get_time_issued = function(Trello_API_token) {

  as.character(attr(Trello_API_token, which = "issued", exact = TRUE))

}

get_time_expiry = function(Trello_API_token) {

  expiration = attr(Trello_API_token, which = "expiration", exact = TRUE)

  if (identical(expiration, "never")) {
    return("never")
  }

  issued = attr(Trello_API_token, which = "issued", exact = TRUE)

  expiry = switch(
    expiration,
    `1hour`  = list( 1, units = "hours"),
    `1day`   = list( 1, units = "days"),
    `30days` = list(30, units = "days"),
    stop("expiration `", expiration, "` not implemented", call. = FALSE)
  )

  expiration_time = issued + do.call(as.difftime, expiry)

  return(paste0(
    as.character(expiration_time),
    if (expiration_time < Sys.time()) " [expired]"
  ))

}
