% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/query.R
\name{query-matches-and-captures}
\alias{query-matches-and-captures}
\alias{query_matches}
\alias{query_captures}
\title{Query matches and captures}
\usage{
query_matches(x, node, ..., range = NULL)

query_captures(x, node, ..., range = NULL)
}
\arguments{
\item{x}{\verb{[tree_sitter_query]}

A query.}

\item{node}{\verb{[tree_sitter_node]}

A node to run the query over.}

\item{...}{These dots are for future extensions and must be empty.}

\item{range}{\verb{[tree_sitter_range / NULL]}

An optional range to restrict the query to.}
}
\description{
These two functions execute a query on a given \code{node}, and return the
captures of the query for further use. Both functions return the same
information, just structured differently depending on your use case.
\itemize{
\item \code{query_matches()} returns the captures first grouped by \emph{pattern}, and
further grouped by \emph{match} within each pattern. This is useful if you
include multiple patterns in your query.
\item \code{query_captures()} returns a flat list of captures ordered by their node
location in the original text. This is normally the easiest structure to
use if you have a single pattern without any alternations that would
benefit from having individual captures split by match.
}

Both also return the capture name, i.e. the \verb{@name} you specified in your
query.
}
\section{Predicates}{


There are 3 core types of predicates supported:
\itemize{
\item \verb{#eq? @capture "string"}
\item \verb{#eq? @capture1 @capture2}
\item \verb{#match? @capture "regex"}
}

Here are a few examples:

\if{html}{\out{<div class="sourceCode">}}\preformatted{# Match an identifier named `"name-of-interest"`
(
  (identifier) @id
  (#eq? @id "name-of-interest")
)

# Match a binary operator where the left and right sides are the same name
(
  (binary_operator
    lhs: (identifier) @id1
    rhs: (identifier) @id2
  )
  (#eq? @id1 @id2)
)

# Match a name with a `_` in it
(
  (identifier) @id
  (#match? @id "_")
)
}\if{html}{\out{</div>}}

Each of these predicates can be inverted with a \verb{not-} prefix.

\if{html}{\out{<div class="sourceCode">}}\preformatted{(
  (identifier) @id
  (#not-eq? @id "name-of-interest")
)
}\if{html}{\out{</div>}}

Each of these predicates can be converted from an \emph{all} style predicate to an
\emph{any} style predicate with an \verb{any-} prefix. This is only useful with
\emph{quantified} captures, i.e. \verb{(comment)+}, where the \code{+} specifies "one or
more comment".

\if{html}{\out{<div class="sourceCode">}}\preformatted{# Finds a block of comments where ALL comments are empty comments
(
  (comment)+ @comment
  (#eq? @comment "#")
)

# Finds a block of comments where ANY comments are empty comments
(
  (comment)+ @comment
  (#any-eq? @comment "#")
)
}\if{html}{\out{</div>}}

This is the full list of possible predicate permutations:
\itemize{
\item \verb{#eq?}
\item \verb{#not-eq?}
\item \verb{#any-eq?}
\item \verb{#any-not-eq?}
\item \verb{#match?}
\item \verb{#not-match?}
\item \verb{#any-match?}
\item \verb{#any-not-match?}
}
\subsection{String double quotes}{

The underlying tree-sitter predicate parser requires that strings supplied
in a query must use double quotes, i.e. \code{"string"} not \code{'string'}. If you
try and use single quotes, you will get a query error.
}

\subsection{\verb{#match?} regex}{

The regex support provided by \verb{#match?} is powered by \code{\link[=grepl]{grepl()}}.

Escapes are a little tricky to get right within these match regex strings.
To use something like \verb{\\s} in the regex string, you need the literal text
\verb{\\\\s} to appear in the string to tell the tree-sitter regex engine to escape
the backslash so you end up with just \verb{\\s} in the captured string. This
requires putting two literal backslash characters in the R string itself,
which can be accomplished with either \code{"\\\\\\\\s"} or using a raw string like
\code{r'["\\\\s"]'} which is typically a little easier. You can also write your
queries in a separate file (typically called \code{queries.scm}) and read them
into R, which is also a little more straightforward because you can just
write something like \verb{(#match? @id "^\\\\s$")} and that will be read in
correctly.
}
}

\examples{
\dontshow{if (rlang::is_installed("treesitter.r")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# ---------------------------------------------------------------------------
# Simple query

text <- "
foo + b + a + ab
and(a)
"

source <- "
(identifier) @id
"

language <- treesitter.r::language()

query <- query(language, source)
parser <- parser(language)
tree <- parser_parse(parser, text)
node <- tree_root_node(tree)

# A flat ordered list of captures, that's most useful here since
# we only have 1 pattern!
captures <- query_captures(query, node)
captures$node

# ---------------------------------------------------------------------------
# Quantified query

text <- "
# this
# that
NULL

# and
# here
1 + 1

# there
2
"

# Find blocks of one or more comments
# The `+` is a regex `+` meaning "one or more" comments in a row
source <- "
(comment)+ @comment
"

language <- treesitter.r::language()

query <- query(language, source)
parser <- parser(language)
tree <- parser_parse(parser, text)
node <- tree_root_node(tree)

# The extra structure provided by `query_matches()` is useful here so
# we can see the 3 distinct blocks of comments
matches <- query_matches(query, node)

# We provided one query pattern, so lets extract that
matches <- matches[[1]]

# 3 blocks of comments
matches[[1]]
matches[[2]]
matches[[3]]

# ---------------------------------------------------------------------------
# Multiple query patterns

# If you know you need to run multiple queries, you can run them all at once
# in one pass over the tree by providing multiple query patterns.

text <- "
a <- 1
b <- function() {}
c <- b
"

# Use an extra set of `()` to separate multiple query patterns
source <- "
(
  (identifier) @id
)
(
  (binary_operator) @binary
)
"

language <- treesitter.r::language()

query <- query(language, source)
parser <- parser(language)
tree <- parser_parse(parser, text)
node <- tree_root_node(tree)

# The extra structure provided by `query_matches()` is useful here so
# we can separate the two queries
matches <- query_matches(query, node)

# First query - all identifiers
matches[[1]]

# Second query - all binary operators
matches[[2]]

# ---------------------------------------------------------------------------
# The `#eq?` and `#match?` predicates

text <- '
fn(a, b)

test_that("this", {
  test
})

fn_name(args)

test_that("that", {
  test
})

fn2_(args)
'

language <- treesitter.r::language()
parser <- parser(language)
tree <- parser_parse(parser, text)
node <- tree_root_node(tree)

# Use an extra set of outer `()` when you are applying a predicate to ensure
# the query pattern is grouped with the query predicate.
# This one finds all function calls where the function name is `test_that`.
source <- '
(
  (call
    function: (identifier) @name
  ) @call
  (#eq? @name "test_that")
)
'

query <- query(language, source)

# It's fine to have a flat list of captures here, but we probably want to
# remove the `@name` captures and just retain the full `@call` captures.
captures <- query_captures(query, node)
captures$node[captures$name == "call"]

# This one finds all functions with a `_` in their name. It uses the R
# level `grepl()` for the regex processing.
source <- '
(
  (call
    function: (identifier) @name
  ) @call
  (#match? @name "_")
)
'

query <- query(language, source)

captures <- query_captures(query, node)
captures$node[captures$name == "call"]

# ---------------------------------------------------------------------------
# The `any-` and `not-` predicate modifiers

text <- '
# 1
#
# 2
NULL

# 3
# 4
NULL

#
#
NULL

#
# 5
#
# 6
#
NULL
'

language <- treesitter.r::language()
parser <- parser(language)
tree <- parser_parse(parser, text)
node <- tree_root_node(tree)

# Two queries:
# - Find comment blocks where there is at least one empty comment
# - Find comment blocks where there is at least one non-empty comment
source <- '
(
  (comment)+ @comment
  (#any-eq? @comment "#")
)
(
  (comment)+ @comment
  (#any-not-eq? @comment "#")
)
'

query <- query(language, source)

matches <- query_matches(query, node)

# Query 1 has 3 comment blocks that match
query1 <- matches[[1]]
query1[[1]]
query1[[2]]
query1[[3]]

# Query 2 has 3 comment blocks that match (a different set than query 1!)
query2 <- matches[[2]]
query2[[1]]
query2[[2]]
query2[[3]]
\dontshow{\}) # examplesIf}
}
