#' Options for the interactive tmap viewer
#' 
#' Set the options for the interactive tmap viewer. Some of these options can also be set with \code{\link{tm_layout}}, since they are style dependent (e.g., the choice of basemaps). The function \code{tm_view} overrides these options when specified.
#' 
#' @param alpha transparency (opacity) parameter applied to whole map. By default, it is set to \code{0.7} if basemaps are used, and \code{1} otherwise.  
#' @param colorNA default color for missing values in interactive mode. If the color of missing values is not defined in the layer functions (e.g. \code{\link{tm_fill}}), then the default color is taken from the \code{na} value of the \code{aes.color} argument in \code{\link{tm_layout}}. This \code{colorNA} argument (if not \code{NA} itself) overrides that default value. For interactive maps, it can be useful to set \code{colorNA} to \code{NULL}, which means transparent.
#' @param projection projection. Either a EPSG number, or a \code{leaflet_crs} object created with \code{\link[leaflet:leafletCRS]{leafletCRS}}. By default, the Web Mercator (3857) is used, since the vast majority of basemaps are rendered accordingly. Other standards are EPSG numbers 4326 (WGS84) and 3395 (Mercator). If set to 0, the projection of the master shape is used (see \code{\link{tm_shape}}) provided that a EPSG number can be extracted.
#' @param symbol.size.fixed should symbol sizes be fixed while zooming?
#' @param dot.size.fixed should dot sizes be fixed while zooming?
#' @param text.size.variable should text size variables be allowed in view mode? By default \code{FALSE}, since in many applications, the main reason to vary text size is to prevent occlusion in plot mode, which is often not a problem in view mode due to the ability to zoom in.
#' @param bbox bounding box. One of the following:
#' \itemize{
#' \item A bounding box (an \code{\link[sf:sf]{sf}} bbox object, see \code{\link[sf:st_bbox]{st_bbox}}, or object that can be read by \code{\link[tmaptools:bb]{bb}}. 
#' \item Open Street Map search query. The bounding is automatically generated by querying \code{q} from Open Street Map Nominatim. See \url{http://wiki.openstreetmap.org/wiki/Nominatim}.}
#' If set, it overrides \code{set.view} and all bbox arguments of \code{\link{tm_shape}}.
#' @param set.bounds logical that determines whether maximum bounds are set, or a numeric vector of four values that specify the lng1, lat1, lng2, and lat2 coordinates (see \code{\link[leaflet:setMaxBounds]{setMaxBounds}}).
#' @param set.view numeric vector that determines the view. Either a vector of three: lng, lat, and zoom, or a single value: zoom. See \code{\link[leaflet:setView]{setView}}. Only applicable if \code{bbox} is not specified
#' @param set.zoom.limits numeric vector of two that set the minimum and maximum zoom levels (see \code{\link[leaflet:tileOptions]{tileOptions}}).
#' @param view.legend.position Character vector of two values, specifying the position of the legend. Use "left" or "right" for the first value and "top" or "bottom" for the second value. It overrides the value of \code{legend.position} of \code{\link{tm_layout}}, unless set to \code{NA}.
#' @param control.position Character vector of two values, specifying the position of the layer control UI. Use "left" or "right" for the first value and "top" or "bottom" for the second value.
#' @param popup.all.data not used anymore. As of version 1.6, the popups are specified by the argument \code{popup.vars} in the layer functions \code{\link{tm_fill}}, \code{\link{tm_symbols}}, and \code{\link{tm_lines}}.
#' @param legend.position not used anymore, renamed to view.legend.position
#' @param basemaps Deprecated. Use \code{\link{tm_basemap}} instead, or set the default basemaps in \code{\link{tmap_options}}
#' @param basemaps.alpha Deprecated. See \code{basemaps}
#' @example ./examples/tm_view.R
#' @seealso \href{../doc/tmap-getstarted.html}{\code{vignette("tmap-getstarted")}}
#' @references Tennekes, M., 2018, {tmap}: Thematic Maps in {R}, Journal of Statistical Software, 84(6), 1-39, \href{https://doi.org/10.18637/jss.v084.i06}{DOI}
#' @export
tm_view <- function(alpha,
					colorNA,
					projection,
					symbol.size.fixed,
					dot.size.fixed,
					text.size.variable,
					bbox,
					set.bounds,
					set.view,
					set.zoom.limits,
					view.legend.position,
					control.position,
					popup.all.data,
					legend.position,
					basemaps = NULL,
					basemaps.alpha = NULL) {
	e1 <- parent.frame()
	args <- lapply(as.list(match.call()[-1]), eval, envir = e1)
	args$style <- NA
	
	g <- list(tm_layout=args)
	class(g) <- "tm"
	g
}
