% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/time_cut.R
\name{time_cut}
\alias{time_cut}
\alias{time_cut_n}
\alias{time_cut_width}
\alias{time_breaks}
\title{Cut dates and datetimes into regularly spaced date or datetime intervals}
\usage{
time_cut(
  x,
  n = 5,
  timespan = NULL,
  from = NULL,
  to = NULL,
  time_floor = FALSE,
  week_start = getOption("lubridate.week.start", 1)
)

time_cut_n(
  x,
  n = 5,
  timespan = NULL,
  from = NULL,
  to = NULL,
  time_floor = FALSE,
  week_start = getOption("lubridate.week.start", 1)
)

time_cut_width(x, timespan = granularity(x), from = NULL, to = NULL)

time_breaks(
  x,
  n = 5,
  timespan = NULL,
  from = NULL,
  to = NULL,
  time_floor = FALSE,
  week_start = getOption("lubridate.week.start", 1)
)
}
\arguments{
\item{x}{Time vector. \cr
E.g. a \code{Date}, \code{POSIXt}, \code{numeric} or any time-based vector.}

\item{n}{Number of breaks.}

\item{timespan}{\link{timespan}.}

\item{from}{Start time.}

\item{to}{End time.}

\item{time_floor}{Logical. Should the initial date/datetime be
floored before building the sequence?}

\item{week_start}{day on which week starts following ISO conventions - 1
means Monday (default), 7 means Sunday.
This is only used when \code{time_floor = TRUE}.}
}
\value{
\code{time_breaks} returns a vector of breaks. \cr
\code{time_cut} returns either a vector or \code{time_interval}. \cr
}
\description{
Useful functions especially for when plotting time-series.
\code{time_cut} makes approximately \code{n} groups of equal time range.
It prioritises the highest time unit possible, making axes look
less cluttered and thus prettier.
\code{time_breaks} returns only the breaks.
}
\details{
To retrieve regular time breaks that simply spans the range of \code{x},
use \code{time_seq()} or \code{time_aggregate()}.
This can also be achieved in \code{time_cut()} by supplying \code{n = Inf}.

By default \code{time_cut()} will try to find
the prettiest way of cutting the interval by
trying to cut the date/date-times into
groups of the highest possible time units,
starting at years and ending at milliseconds.

When \code{x} is a numeric vector, \code{time_cut} will behave similar to \code{time_cut}
except for 3 things:
\itemize{
\item The intervals are all right-open and of equal width.
\item The left value of the leftmost interval is always \code{min(x)}.
\item Up to \code{n} breaks are created, i.e \verb{<= n} breaks. This is to prioritise
pretty breaks.
}
}
\examples{
library(timeplyr)
library(fastplyr)
library(cheapr)
library(lubridate)
library(ggplot2)
library(dplyr)

time_cut_n(1:10, n = 5)

# Easily create custom time breaks
df <- nycflights13::flights \%>\%
  f_slice_sample(n = 100) \%>\%
  with_local_seed(.seed = 8192821) \%>\%
  f_select(time_hour) \%>\%
  fastplyr::f_arrange(time_hour) \%>\%
  mutate(date = as_date(time_hour))

# time_cut_n() and time_breaks() automatically find a
# suitable way to cut the data
time_cut_n(df$date) \%>\%
  interval_count()
# Works with datetimes as well
time_cut_n(df$time_hour, n = 5) \%>\%
  interval_count()
time_cut_n(df$date, timespan = "month") \%>\%
  interval_count()
# Just the breaks
time_breaks(df$date, n = 5, timespan = "month")

cut_dates <- time_cut_n(df$date)
date_breaks <- time_breaks(df$date)

# When n = Inf it should be equivalent to using time_cut_width
identical(time_cut_n(df$date, n = Inf, "month"),
          time_cut_width(df$date, "month"))
# To get exact breaks at regular intervals, use time_grid
weekly_breaks <- time_grid(
  df$date, "5 weeks",
  from = floor_date(min(df$date), "week", week_start = 1)
)
weekly_labels <- format(weekly_breaks, "\%b-\%d")
df \%>\%
  time_by(date, "week", .name = "date") \%>\%
  f_count() \%>\%
  mutate(date = interval_start(date)) \%>\%
  ggplot(aes(x = date, y = n)) +
  geom_bar(stat = "identity") +
  scale_x_date(breaks = weekly_breaks,
               labels = weekly_labels)
}
