% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EstimationBR.R
\name{EstimationBR}
\alias{EstimationBR}
\title{Estimation of the parameters of the Brown-Resnick process}
\usage{
EstimationBR(x, locations, indices, k, method, isotropic = FALSE,
  biascorr = FALSE, Tol = 1e-05, k1 = (nrow(x) - 10), tau = 5,
  startingValue = NULL, Omega = diag(nrow(indices)), iterate = FALSE,
  covMat = TRUE)
}
\arguments{
\item{x}{An \eqn{n} x \eqn{d} data matrix.}

\item{locations}{A \eqn{d} x 2 matrix containing the Cartesian coordinates of \eqn{d} points in the plane.}

\item{indices}{A \eqn{q} x \eqn{d} matrix containing exactly 2 ones per row, representing a pair of points from the matrix \code{locations}, and zeroes elsewhere.}

\item{k}{An integer between 1 and \eqn{n - 1}; the threshold parameter in the definition of the empirical stable tail dependence function.}

\item{method}{Choose between \code{Mestimator} and \code{WLS}.}

\item{isotropic}{A Boolean variable. If \code{FALSE} (the default), then an anisotropic process is estimated.}

\item{biascorr}{For \code{method = "WLS"} only. If \code{TRUE}, then the bias-corrected estimator of the stable tail dependence function is used. Defaults to \code{FALSE}.}

\item{Tol}{For \code{method = "Mestimator"} only. The tolerance parameter used in the numerical integration procedure. Defaults to 1e-05.}

\item{k1}{For \code{biascorr = TRUE} only. The value of \eqn{k_1} in the definition of the bias-corrected estimator of the stable tail dependence function.}

\item{tau}{For \code{biascorr = TRUE} only. The parameter of the power kernel.}

\item{startingValue}{Initial value of the parameters in the minimization routine. Defaults to \eqn{c(1,1.5)} if \code{isotropic = TRUE} and \eqn{c(1, 1.5, 0.75, 0.75)} if \code{isotropic = FALSE}.}

\item{Omega}{A \eqn{q} x \eqn{q} matrix specifying the metric with which the distance between the parametric and nonparametric estimates will be computed. The default is the identity matrix, i.e., the Euclidean metric.}

\item{iterate}{A Boolean variable. If \code{TRUE}, then for \code{method = "Mestimator"} the estimator is calculated twice, first with \code{Omega} specified by the user, and then a second time with the optimal \code{Omega} calculated at the initial estimate. If \code{method = "WLS"}, then continuous updating is used.}

\item{covMat}{A Boolean variable. If \code{TRUE} (the default), the covariance matrix is calculated. Standard errors are obtained by taking the square root of the diagonal elements.}
}
\value{
A list with the following components:
\tabular{ll}{
\code{theta} \tab The estimator using the optimal weight matrix. \cr
\code{theta_pilot} \tab The estimator without the optimal weight matrix.\cr
\code{covMatrix} \tab The estimated covariance matrix for the estimator. \cr
\code{value} \tab The value of the minimized function at \code{theta}. \cr
}
}
\description{
Estimation the parameters of the Brown-Resnick process, using either the pairwise M-estimator or weighted least squares (WLS).
}
\details{
The parameters of the Brown-Resnick process are either \eqn{(\alpha,\rho)} for an isotropic process or \eqn{(\alpha,\rho,\beta,c)} for an anisotropic process. The matrix \code{indices} can be either user-defined or returned from the function \code{selectGrid} with \code{cst = c(0,1)}. Estimation might be rather slow when \code{iterate = TRUE} or even when \code{covMat = TRUE}.
}
\examples{
## define the locations of 9 stations
## locations <- cbind(rep(c(1:3), each = 3), rep(1:3, 3))
## select the pairs of locations
## indices <- selectGrid(cst = c(0,1), d = 9, locations = locations, maxDistance = 1.5)
## The Brown-Resnick process
## set.seed(1)
## x <- SpatialExtremes::rmaxstab(n = 1000, coord = locations, cov.mod = "brown",
##                               range = 3, smooth = 1)
## Calculate the estimtors.
## EstimationBR(x, locations, indices, 100, method = "Mestimator", isotropic = TRUE,
##             covMat = FALSE)$theta
## EstimationBR(x, locations, indices, 100, method = "WLS", isotropic = TRUE,
## covMat = FALSE)$theta
}
\references{
Einmahl, J.H.J., Kiriliouk, A., and Segers, J. (2016). A continuous updating weighted least squares estimator of tail dependence in high dimensions. See http://arxiv.org/abs/1601.04826.

Einmahl, J.H.J., Kiriliouk, A., Krajina, A., and Segers, J. (2016). An Mestimator of spatial tail dependence. Journal of the Royal Statistical Society: Series B (Statistical Methodology), 78(1), 275-298.
}
\seealso{
\code{\link{selectGrid}}
}

