% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/chow.test.R
\name{chow.test}
\alias{chow.test}
\title{Chow Test for Structural Break}
\usage{
chow.test(x, SB, nboot = 500, start = NULL, end = NULL,
  frequency = NULL, format = NULL, dateVector = NULL)
}
\arguments{
\item{x}{An object of class 'vars', 'vec2var', 'nlVar'. Estimated VAR object. Or an object of class 'chowpretest' from stability()}

\item{SB}{Integer, vector or date character. The structural break is specified either by an integer (number of observations in the pre-break period),
a vector of ts() frequencies if a ts object is used in the VAR or a date character. If a date character is provided, either a date vector containing the whole time line
in the corresponding format or common time parameters need to be provided}

\item{nboot}{Integer. Number of bootstrap iterations to calculate quantiles and p-values}

\item{start}{Character. Start of the time series (only if dateVector is empty)}

\item{end}{Character. End of the time series (only if dateVector is empty)}

\item{frequency}{Character. Frequency of the time series (only if dateVector is empty)}

\item{format}{Character. Date format (only if dateVector is empty)}

\item{dateVector}{Vector. Vector of time periods containing SB in corresponding format}
}
\value{
A list of class "chow" with elements
\item{lambda_bp}{Test statistic of the Chow test with break point}
\item{testcrit_bp}{Critical value of the test statistic lambda_bp}
\item{p.value_bp}{p-value of the test statistic lambda_bp}
\item{lambda_sp}{Test statistic of the Chow test with sample split}
\item{testcrit_sp}{Critical value of the test statistic lambda_sp}
\item{p.value_sp}{p-value of the test statistic lambda_sp}
\item{SB}{Structural break tested}
\item{SBcharacter}{Structural break tested as character}
\item{p}{Number of lags used}
}
\description{
The Chow test for structural change is implemented as sample-split and break-point test (see Luetkepohl and Kraetzig, 2004, p. 135). An estimated VAR model and the presupposed structural break need to be provided.
}
\examples{
\donttest{
# Testing for structural break in USA data
#' # data contains quartlery observations from 1965Q1 to 2008Q2
# assumed structural break in 1979Q3
# x = output gap
# pi = inflation
# i = interest rates
set.seed(23211)
v1 <- vars::VAR(USA, lag.max = 10, ic = "AIC" )
z1 <- chow.test(v1, SB = 59)
summary(z1)

#Using stability() to find potential break point and sample split
x1 <- stability(v1, type = "mv-chow-test")
plot(x1)
z1.1 <- chow.test(x1)
summary(z1.1)
#Or using sample split as benchmark
x1$break_point <- FALSE
z1.1 <- chow.test(x1)
summary(z1.1)

#Structural brake via Dates
#given that time series vector with dates is available
dateVector <- seq(as.Date("1965/1/1"), as.Date("2008/7/1"), "quarter")
z2 <- chow.test(v1, SB = "1979-07-01", format = "\%Y-\%m-\%d", dateVector = dateVector)
summary(z2)

# alternatively pass sequence arguments directly
z3 <- chow.test(v1, SB = "1979-07-01", format = "\%Y-\%m-\%d",
                start = "1965-01-01", end = "2008-07-01",
                frequency = "quarter")
summary(z3)

# or provide ts date format (For quarterly, monthly, weekly and daily frequencies only)
z4 <- chow.test(v1, SB = c(1979,3))
summary(z4)
}
}
\references{
Luetkepohl, H., 2005. New introduction to multiple time series analysis, Springer-Verlag, Berlin.\cr
     Luetkepohl, H., Kraetzig, M., 2004. Applied time series econometrics, Cambridge University Press, Cambridge.
}
\seealso{
\code{\link{stability}}
}
