% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/recalib.R
\name{recalib}
\alias{recalib}
\title{Calibrate weights}
\usage{
recalib(
  dat,
  hid = attr(dat, "hid"),
  weights = attr(dat, "weights"),
  b.rep = attr(dat, "b.rep"),
  period = attr(dat, "period"),
  conP.var = NULL,
  conH.var = NULL,
  epsP = 0.01,
  epsH = 0.02,
  ...
)
}
\arguments{
\item{dat}{either data.frame or data.table containing the sample survey for
various periods.}

\item{hid}{character specifying the name of the column in \code{dat} containing
the household ID.}

\item{weights}{character specifying the name of the column in \code{dat}
containing the sample weights.}

\item{b.rep}{character specifying the names of the columns in \code{dat}
containing bootstrap weights which should be recalibratet}

\item{period}{character specifying the name of the column in \code{dat} containing
the sample period.}

\item{conP.var}{character vector containig person-specific variables to which
weights should be calibrated or a list of such character vectors.
Contingency tables for the population are calculated per \code{period} using
\code{weights}.}

\item{conH.var}{character vector containig household-specific variables to
which weights should be calibrated or a list of such character vectors.
Contingency tables for the population are calculated per \code{period} using
\code{weights}.}

\item{epsP}{numeric value specifying the convergence limit for \code{conP.var}
or \code{conP}, see \code{\link[=ipf]{ipf()}}.}

\item{epsH}{numeric value specifying the convergence limit for \code{conH.var}
or \code{conH}, see \code{\link[=ipf]{ipf()}}.}

\item{...}{additional arguments passed on to function \code{\link[=ipf]{ipf()}} from this
package.}
}
\value{
Returns a data.table containing the survey data as well as the
calibrated weights for the bootstrap replicates. The original bootstrap
replicates are overwritten by the calibrated weights. If calibration of a
bootstrap replicate does not converge the bootsrap weight is not returned
and numeration of the returned bootstrap weights is reduced by one.
}
\description{
Calibrate weights for bootstrap replicates by using iterative proportional
updating to match population totals on various household and personal levels.
}
\details{
\code{recalib} takes survey data (\code{dat}) containing the bootstrap replicates
generated by \link{draw.bootstrap} and calibrates weights for each bootstrap
replication according to population totals for person- or household-specific
variables. \cr
\code{dat} must be household data where household members correspond to multiple
rows with the same household identifier. The data should at least containt
the following columns:
\itemize{
\item Column indicating the sample period;
\item Column indicating the household ID;
\item Column containing the household sample weights;
\item Columns which contain the bootstrap replicates (see output of
\link{draw.bootstrap});
\item Columns indicating person- or household-specific variables for which sample
weight should be adjusted.
}

For each period and each variable in \code{conP.var} and/or \code{conH.var} contingency
tables are estimated to get margin totals on personal- and/or
household-specific variables in the population.\cr
Afterwards the bootstrap replicates are multiplied with the original sample
weight and the resulting product ist then adjusted using \code{\link[=ipf]{ipf()}} to match the
previously calcualted contingency tables. In this process the columns of the
bootstrap replicates are overwritten by the calibrated weights.\cr
}
\examples{
\dontrun{

eusilc <- demo.eusilc(prettyNames = TRUE)

dat_boot <- draw.bootstrap(eusilc, REP = 10, hid = "hid",
                           weights = "pWeight",
                           strata = "region", period = "year")

# calibrate weight for bootstrap replicates
dat_boot_calib <- recalib(dat_boot, conP.var = "gender", conH.var = "region",
                          verbose = TRUE)


# calibrate on other variables
dat_boot_calib <- recalib(dat_boot, conP.var = c("gender", "age"),
                          conH.var = c("region", "hsize"), verbose = TRUE)

# supply contingency tables directly
conP <- xtabs(pWeight ~ age + gender + year, data = eusilc)
conH <- xtabs(pWeight ~ hsize + region + year,
              data = eusilc[!duplicated(paste(db030,year))])
dat_boot_calib <- recalib(dat_boot, conP.var = NULL,
                          conH.var = NULL, conP = list(conP),
                          conH = list(conH), verbose = TRUE)
}

}
\seealso{
\code{\link[=ipf]{ipf()}} for more information on iterative
proportional fitting.
}
\author{
Johannes Gussenbauer, Alexander Kowarik, Statistics Austria
}
