% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Democratic.R
\name{democratic}
\alias{democratic}
\title{Democratic method}
\usage{
democratic(x, y, x.inst = TRUE, learners, learners.pars = NULL,
  preds = rep("predict", length(learners)), preds.pars = NULL)
}
\arguments{
\item{x}{A object that can be coerced as matrix. This object has two possible 
interpretations according to the value set in the \code{x.inst} argument:
a matrix with the training instances where each row represents a single instance
or a precomputed (distance or kernel) matrix between the training examples.}

\item{y}{A vector with the labels of the training instances. In this vector 
the unlabeled instances are specified with the value \code{NA}.}

\item{x.inst}{A boolean value that indicates if \code{x} is or not an instance matrix.
Default is \code{TRUE}.}

\item{learners}{A list of functions or strings naming the functions for 
training the different supervised base classifiers.}

\item{learners.pars}{A list with the set of additional parameters for each
learner functions if necessary.
Default is \code{NULL}.}

\item{preds}{A list of functions or strings naming the functions for
predicting the probabilities per classes,
using the base classifiers trained with the functions defined in \code{learners}.
Default is \code{"predict"} function for each learner in \code{learners}.}

\item{preds.pars}{A list with the set of additional parameters for each
function in \code{preds} if necessary.
Default is \code{NULL}.}
}
\value{
A list object of class "democratic" containing:
\describe{
  \item{W}{A vector with the confidence-weighted vote assigned to each classifier.}
  \item{model}{A list with the final N base classifiers trained using the 
  enlarged labeled set.}
  \item{model.index}{List of N vectors of indexes related to the training instances 
  used per each classifier. These indexes are relative to the \code{y} argument.}
  \item{instances.index}{The indexes of all training instances used to
  train the N \code{models}. These indexes include the initial labeled instances
  and the newly labeled instances. These indexes are relative to the \code{y} argument.}
  \item{model.index.map}{List of three vectors with the same information in \code{model.index}
  but the indexes are relative to \code{instances.index} vector.}
  \item{classes}{The levels of \code{y} factor.}
  \item{preds}{The functions provided in the \code{preds} argument.}
  \item{preds.pars}{The set of lists provided in the \code{preds.pars} argument.}
  \item{x.inst}{The value provided in the \code{x.inst} argument.}
}
}
\description{
Democratic Co-Learning is a semi-supervised learning algorithm with a 
co-training style. This algorithm trains N classifiers with different learning schemes 
defined in list \code{gen.learners}. During the iterative process, the multiple classifiers
with different inductive biases label data for each other.
}
\details{
This method trains an ensemble of diverse classifiers. To promote the initial diversity 
the classifiers must represent different learning schemes.
When x.inst is \code{FALSE} all \code{learners} defined must be able to learn a classifier
from the precomputed matrix in \code{x}.
The iteration process of the algorithm ends when no changes occurs in 
any model during a complete iteration.
The generation of the final hypothesis is 
produced via a weigthed majority voting.
}
\examples{

library(ssc)

## Load Wine data set
data(wine)

cls <- which(colnames(wine) == "Wine")
x <- wine[, -cls] # instances without classes
y <- wine[, cls] # the classes
x <- scale(x) # scale the attributes

## Prepare data
set.seed(20)
# Use 50\% of instances for training
tra.idx <- sample(x = length(y), size = ceiling(length(y) * 0.5))
xtrain <- x[tra.idx,] # training instances
ytrain <- y[tra.idx]  # classes of training instances
# Use 70\% of train instances as unlabeled set
tra.na.idx <- sample(x = length(tra.idx), size = ceiling(length(tra.idx) * 0.7))
ytrain[tra.na.idx] <- NA # remove class information of unlabeled instances

# Use the other 50\% of instances for inductive testing
tst.idx <- setdiff(1:length(y), tra.idx)
xitest <- x[tst.idx,] # testing instances
yitest <- y[tst.idx] # classes of testing instances

## Example: Training from a set of instances with 
# 1-NN and C-svc (SVM) as base classifiers.
# knn3 learner
library(caret)
knn <- knn3             # learner function
knn.pars <- list(k = 1) # parameters for learner function
knn.prob <- predict     # function to predict probabilities
knn.prob.pars <- NULL   # parameters for prediction function

# ksvm learner
library(kernlab)
svm <- ksvm             # learner function
svm.pars <- list(       # parameters for learner function
  type = "C-svc",  C = 1, 
  kernel = "rbfdot", kpar = list(sigma = 0.048),
  prob.model = TRUE,
  scaled = FALSE
)
svm.prob <- predict     # function to predict probabilities
svm.prob.pars <- list(  # parameters for prediction function
  type = "probabilities"
)

# train a model
m <- democratic(x = xtrain, y = ytrain, 
                learners = list(knn, svm), 
                learners.pars = list(knn.pars, svm.pars), 
                preds = list(knn.prob, svm.prob), 
                preds.pars = list(knn.prob.pars, svm.prob.pars))
# predict classes
m.pred <- predict(m, xitest)
table(m.pred, yitest)

}
